import { faBook, faCheck } from '@fortawesome/free-solid-svg-icons';
import { faPlus } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Dictionary } from '@reduxjs/toolkit';
import { Button } from '@yandex-cloud/uikit';
import {
   EMPTY_VALUE,
   EmptyContainer,
   EmptyContainerType,
   ExternalLink,
   FormButton,
   Hint,
} from '@yandex-infracloud-ui/libs';
import React, { useMemo } from 'react';
// noinspection ES6PreferShortImport
import { HierarchicalTable, HierarchicalTableConfig, LevelConfig } from '../../../../../components/lib';
import { EXTERNAL_LINKS } from '../../../../../models';
import { DeployUnitSecret, DeployUnitSecretVersion, Secret, SecretVersion } from '../../../models';
import { AliasHint } from '../../hints';
import { SecretListLayoutActionsCell } from '../__actionsCell/SecretListLayoutActionsCell';
import { SecretListLayoutAliasCell } from '../__aliasCell/SecretListLayoutAliasCell';
import { SecretListLayoutSecretCell } from '../__secretCell/SecretListLayoutSecretCell';
import { SecretListLayoutVersionCell } from '../__versionCell/SecretListLayoutVersionCell';
import { SecretListLayoutContext } from '../models';

import classes from './SecretListLayout.module.css';

const secretCellClassNameFn = (secret: DeployUnitSecret) =>
   secret.versions.every(v => v.removed) ? classes.removedCell : '';

const versionCellClassNameFn = (version: DeployUnitSecretVersion) => (version.removed ? classes.removedCell : '');

const config: HierarchicalTableConfig = {
   addCheckbox: false, // TODO когда появится поддержка пакетных операций - разрешить выбирать
   levels: [
      {
         id: 'secret',
         getId: v => v.secretUuid,
         getChildren: v => v.versions,
         columns: [
            {
               id: 'secret',
               head: 'Secret',
               className: classes.secretCol,
               cellClassNameFn: secretCellClassNameFn,
               render: (secret, parent, context) => <SecretListLayoutSecretCell secret={secret} context={context} />,
            },
         ],
      } as LevelConfig<void, DeployUnitSecret, DeployUnitSecretVersion, SecretListLayoutContext>,
      {
         id: 'version',
         getId: v => v.alias,
         columns: [
            {
               id: 'version',
               head: 'Version',
               className: classes.versionCol,
               cellClassNameFn: versionCellClassNameFn,
               render: (version, parent, context) => (
                  <SecretListLayoutVersionCell secret={parent!} version={version} context={context} />
               ),
            },
            {
               id: 'alias',
               head: (
                  <>
                     Alias <Hint text={<AliasHint />} />
                  </>
               ),
               className: classes.aliasCol,
               cellClassNameFn: versionCellClassNameFn,
               render: (version, parent) => <SecretListLayoutAliasCell secret={parent!} version={version} />,
            },
            {
               id: 'token',
               head: 'Token',
               className: classes.tokenCol,
               cellClassName: classes.tokenCell,
               cellClassNameFn: versionCellClassNameFn,
               render: version =>
                  version.token ? (
                     <span title={version.token}>
                        <FontAwesomeIcon icon={faCheck} color={`var(--yc-color-infographics-positive-heavy)`} />
                     </span>
                  ) : (
                     EMPTY_VALUE
                  ),
            },
            {
               id: 'actions',
               head: 'Actions',
               className: classes.actionsCol,
               isHidden: context => context.readonly,
               render: (version, parent, context) => (
                  <SecretListLayoutActionsCell version={version} secret={parent!} context={context} />
               ),
            },
         ],
      } as LevelConfig<DeployUnitSecret, DeployUnitSecretVersion, void, SecretListLayoutContext>,
   ],
};

interface Props {
   disabled: boolean;
   duSecrets: DeployUnitSecret[] | undefined;
   readonly: boolean;
   secrets: Dictionary<Secret | undefined>;
   versions: Dictionary<SecretVersion[]>;

   onAddSecret(): void;

   onAliasEdit(secretUuid: string, duVersion: DeployUnitSecretVersion): void;

   onAddSecretVersion(secretUuid: string): void;

   onMigrate(alias: string): void;

   onMigrateAll(): void;

   onRemoveUnused(secretUuid: string): void;

   onUndoMigration(alias: string): void;
}

export const SecretListLayout: React.FC<Props> = React.memo(
   ({
      disabled,
      duSecrets,
      onAddSecret,
      onAddSecretVersion,
      onAliasEdit,
      onMigrate,
      onMigrateAll,
      onRemoveUnused,
      onUndoMigration,
      readonly,
      secrets,
      versions,
   }) => {
      const context: SecretListLayoutContext = useMemo(
         () => ({
            disabled,
            readonly,
            secrets,
            versions,

            handleAddVersion: onAddSecretVersion,
            handleAliasEdit: onAliasEdit,
            handleMigrate: onMigrate,
            handleRemoveUnused: onRemoveUnused,
            handleUndoMigration: onUndoMigration,
         }),
         [
            disabled,
            onAddSecretVersion,
            onAliasEdit,
            onMigrate,
            onRemoveUnused,
            onUndoMigration,
            readonly,
            secrets,
            versions,
         ],
      );

      if (!duSecrets || duSecrets.length === 0) {
         return (
            <EmptyContainer
               type={EmptyContainerType.EmptyState}
               title={"This DU doesn't use any secret"}
               description={
                  !readonly && (
                     <Button
                        view={'action'}
                        disabled={disabled}
                        onClick={onAddSecret}
                        qa={'SecretList:AddSecretButton'}
                     >
                        Add secret
                     </Button>
                  )
               }
            />
         );
      }

      const hasLegacySecretVersions = duSecrets?.some(s => s.versions.some(v => v.legacy && !v.migrated));

      return (
         <>
            <HierarchicalTable
               className={classes.wrapper}
               config={config}
               context={context}
               disabled={disabled}
               readonly={readonly}
               values={duSecrets}
               data-e2e={'SecretList'}
            />
            {!readonly && (
               <span className={classes.listActions}>
                  <FormButton
                     disabled={disabled}
                     onClick={onAddSecret}
                     skipLeftSpace={true}
                     icon={faPlus}
                     dataE2e={'SecretList:AddSecretButton'}
                  >
                     Add secret
                  </FormButton>

                  {hasLegacySecretVersions && (
                     <FormButton disabled={disabled} onClick={onMigrateAll} dataE2e={'SecretList:MigrateAllButton'}>
                        Migrate all secrets to the new format
                     </FormButton>
                  )}
               </span>
            )}

            <ExternalLink href={EXTERNAL_LINKS.deployDocs.secrets} data-e2e={'SecretList:DocsLink'}>
               <FontAwesomeIcon icon={faBook} /> Secrets in deploy docs
            </ExternalLink>
         </>
      );
   },
);

SecretListLayout.displayName = 'SecretListLayout';
