import { faPlus, faTimes } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { Button } from '@yandex-cloud/uikit';
import {
   classNames,
   DISMISS_REASON,
   EMPTY_VALUE,
   FormButton,
   isEmpty,
   modalService,
   splitBy,
   toasts,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { useField } from 'formik';
import React, { useCallback, useEffect, useMemo } from 'react';
import { useDispatch, useStore } from 'react-redux';
import { takeUntil } from 'rxjs/operators';
import { isMultisecret, LinkToSecret } from '../../../../models/ui/secrets';
import { noop } from '../../../../utils';
import { addSecretContext } from '../../hocs';
import { useDuSecretsWithVersions } from '../../hooks';
import { DeployUnitSecretVersion, SecretsContext, SecretVersion } from '../../models';
import { loadSecrets, RootStateWithSecrets, selectSecretVersions, upsertDeployUnitSecretVersion } from '../../slice';
import { AddSecretModal } from '../AddSecretModal/AddSecretModal';
import { SelectedSecretName } from '../SelectedSecretName/SelectedSecretName';
import { SecretSubFormKeySelect } from './__keySelect/SecretSubFormKeySelect';
import { SecretSubFormVersionSelect } from './__versionSelect/SecretSubFormVersionSelect';

import classes from './SecretSubForm.module.css';

interface Props {
   className?: string;
   duId: string;
   name: string;
   stageId: string;
   readonly?: boolean;
   disabled?: boolean;
   layout: 'vertical' | 'horizontal';
   allowMultisecret?: boolean;

   onChange?(): void;
}

export const SecretSubForm: React.FC<Props> = React.memo(
   ({ className, name, stageId, duId, disabled, readonly, layout, onChange, allowMultisecret = false }) => {
      const store = useStore<RootStateWithSecrets>();
      const dispatch = useDispatch();
      const dismounted = useDismounted();

      const [field, , helpers] = useField<LinkToSecret | null>(name);
      const { duSecrets, versions, secrets } = useDuSecretsWithVersions(stageId, duId);

      const getVersionDetails = useCallback(
         (alias: string) => {
            let duVersion: DeployUnitSecretVersion | null = null;
            let version: SecretVersion | null = null;

            for (const duSecret of duSecrets ?? []) {
               duVersion = duSecret.versions.find(v => v.alias === alias) ?? null;
               if (duVersion) {
                  const duVersionUuid = duVersion.versionUuid;
                  version = versions[duSecret.secretUuid]?.find(v => v.version === duVersionUuid) ?? null;
                  break;
               }
            }

            return { version, duVersion };
         },
         [duSecrets, versions],
      );

      /**
       * Список доступных для выбора ключей
       *
       * Декларативно извлекается из текущий версии секрета
       */
      const availableKeys = useMemo(() => {
         if (!field.value?.alias) {
            return [];
         }

         const { version } = getVersionDetails(field.value.alias);

         return (version?.keys ?? []).filter(k => !isMultisecret(k));
      }, [field.value?.alias, getVersionDetails]);

      const handleVersionChange = useCallback(
         (alias: string) => {
            const { version, duVersion } = getVersionDetails(alias);
            const currentKey = field.value?.key ?? '';
            const [, keys] = splitBy(version?.keys ?? [], isMultisecret);

            helpers.setValue(
               duVersion
                  ? {
                       key: keys.includes(currentKey)
                          ? currentKey
                          : !allowMultisecret && keys.length === 1
                          ? keys[0]
                          : '',
                       alias: duVersion.alias,
                    }
                  : null,
            );

            onChange?.();
         },
         [allowMultisecret, field.value?.key, getVersionDetails, helpers, onChange],
      );

      const handleKeyChange = useCallback(
         (v: string) => {
            helpers.setValue({ ...field.value!, key: v });

            onChange?.();
         },
         [field.value, helpers, onChange],
      );

      const handleClear = useCallback(() => {
         helpers.setValue(null);

         onChange?.();
      }, [helpers, onChange]);

      const handleAddVersion = useCallback(() => {
         modalService
            .open(
               addSecretContext(AddSecretModal, stageId, duId),
               {},
               { closeOnOutsideClick: false, dialogProps: { size: 'm' } },
            )
            .pipe(takeUntil(dismounted))
            .subscribe(
               record => {
                  dispatch(upsertDeployUnitSecretVersion({ stageId, duId, ...record }));

                  // Оставляет текущий key по возможности (если такой есть в новой версии).
                  // Если не получается, то выбирает единственный или стирает ключ, если вариантов много.
                  const secretVersions = selectSecretVersions(store.getState(), record.secretUuid);
                  const version = secretVersions?.find(v => v.version === record.versionUuid);
                  const currentKey = field.value?.key ?? '';
                  const key = version?.keys?.some(k => k === currentKey)
                     ? currentKey
                     : version?.keys?.length === 1
                     ? version.keys[0]
                     : '';

                  helpers.setValue({
                     alias: record.alias,
                     key,
                  });

                  onChange?.();
               },
               reason => {
                  if (reason !== DISMISS_REASON) {
                     toasts.error(reason.toString(), 'Unknown error on adding secret');
                  }
               },
            );
      }, [dismounted, dispatch, onChange, duId, field.value?.key, helpers, stageId, store]);

      useEffect(() => {
         // Загрузка данных о секретах (обогащение)
         const secretUuids = duSecrets?.map(s => s.secretUuid) ?? [];
         if (secretUuids.length > 0) {
            dispatch(loadSecrets(secretUuids));
         }
      }, [dispatch, duSecrets]);

      // region render
      if (readonly) {
         return (
            <SecretsContext.Provider value={{ stageId, duId, reloadUsages: noop }}>
               <div className={classes.readonly}>
                  {field.value ? <SelectedSecretName selectedSecret={field.value} /> : <>{EMPTY_VALUE}</>}
               </div>
            </SecretsContext.Provider>
         );
      }

      const versionField = (
         <SecretSubFormVersionSelect
            dataTest={'SecretSubForm-version'}
            disabled={disabled}
            duSecrets={duSecrets}
            name={`${name}.version`}
            onChange={handleVersionChange}
            secrets={secrets}
            value={field.value?.alias ?? ''}
            versions={versions}
         />
      );

      const keyPlaceholder = field.value?.alias
         ? availableKeys.length > 0
            ? 'Select key'
            : 'Keys not found, access denied?'
         : 'Select version before';

      const keyField = (
         <SecretSubFormKeySelect
            availableKeys={availableKeys}
            dataTest={'SecretSubForm-key'}
            disabled={disabled || !field.value?.alias}
            name={`${name}.key`}
            onChange={handleKeyChange}
            placeholder={keyPlaceholder}
            value={field.value?.key}
            allowMultisecret={allowMultisecret}
         />
      );

      const clearButton = (
         <Button
            size={'s'}
            view={'flat'}
            onClick={handleClear}
            disabled={disabled || !field.value}
            qa={'SecretSubForm:ClearUnused'}
            className={classes.clearButton}
         >
            <FontAwesomeIcon icon={faTimes} />
         </Button>
      );

      const addButton = (
         <FormButton
            className={classes.addButton}
            disabled={disabled}
            onClick={handleAddVersion}
            skipLeftSpace={true}
            dataE2e={'SecretSubForm:AddVersion'}
         >
            <FontAwesomeIcon icon={faPlus} /> Add secret
         </FormButton>
      );

      const showFields = field.value || !isEmpty(duSecrets);

      if (layout === 'vertical') {
         return (
            <div className={className}>
               {showFields ? (
                  <div className={classes.vertical}>
                     <div className={classes.fields}>
                        {versionField} {keyField}
                     </div>

                     {clearButton}
                  </div>
               ) : null}

               {addButton}
            </div>
         );
      }

      return (
         <div className={classNames(classes.horizontal, className)}>
            {showFields && (
               <div className={classes.fields}>
                  {versionField} <span>/</span> {keyField} {clearButton}
               </div>
            )}

            {addButton}
         </div>
      );
      // endregion
   },
);

SecretSubForm.displayName = 'SecretSubForm';
