import { YCSelect, YCSelectItemsGroup } from '@yandex-data-ui/common';
import { BaseInputProps, classNames, EMPTY_VALUE, formatDate, sortHandler } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useMemo } from 'react';

import { EXTERNAL_LINKS } from '../../../../../models';
import { isSecretAliasGenerated } from '../../../helpers';
import { DuSecretsWithVersions } from '../../../hooks';

interface Props extends BaseInputProps<string>, DuSecretsWithVersions {
   dataTest?: string;
}

/**
 * Поле для выбора секрета из списка использованных в DU (указанных в podAgent)
 *
 * Значение поля - это алиас секрета из DU
 */
export const SecretSubFormVersionSelect: React.FC<Props> = React.memo(
   ({
      className,
      dataTest,
      disabled,
      duSecrets,
      onBlur,
      onChange,
      placeholder = 'Select secret from deploy unit',
      readonly,
      readonlyClassName,
      secrets,
      value,
      versions,
   }) => {
      const items: YCSelectItemsGroup[] = useMemo(
         () =>
            duSecrets?.map(duSecret => {
               const secret = secrets[duSecret.secretUuid];

               const groupTitle = secret?.name ?? duSecret.secretUuid;
               return {
                  groupTitle,
                  items: duSecret.versions
                     .filter(duVersion => !duVersion.removed)
                     .map(duVersion => {
                        const version = versions[duSecret.secretUuid]?.find(v => v.version === duVersion.versionUuid);
                        const versionUuid = version?.version ?? duVersion.versionUuid;
                        const versionDate = version?.createdAt ? formatDate(new Date(version.createdAt)) : undefined;

                        const title = isSecretAliasGenerated(duVersion.alias)
                           ? `${groupTitle} / ${versionDate ?? versionUuid}`
                           : `${duVersion.alias}${versionDate ? ` / ${versionDate}` : ''}`;

                        return {
                           key: versionUuid,
                           title,
                           url: EXTERNAL_LINKS.secretVersion(duSecret.secretUuid, versionUuid),
                           value: duVersion.alias,
                           createdAt: version?.createdAt,
                        };
                     })
                     .sort((a, b) => sortHandler(b.createdAt, a.createdAt)),
               };
            }) ?? [],
         [secrets, duSecrets, versions],
      );

      const handleChange = useCallback(
         v => {
            onChange(v);
            onBlur?.();
         },
         [onBlur, onChange],
      );

      // region render
      if (readonly) {
         return <div className={readonlyClassName}>{value || EMPTY_VALUE}</div>;
      }

      return (
         <YCSelect
            applyOnOutsideClick={false}
            className={classNames(className, `test-${dataTest}`)}
            disabled={disabled}
            items={items}
            onUpdate={handleChange}
            placeholder={placeholder}
            showSearch={false}
            type={YCSelect.SINGLE}
            value={value}
         />
      );
      // endregion
   },
);

SecretSubFormVersionSelect.displayName = 'SecretSubFormVersionSelect';
