import { YCSelectItem } from '@yandex-data-ui/common';
import { ExtendedFieldConfig, formatDate, isEmpty, useExtendedField } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useEffect, useMemo } from 'react';
import { useDispatch, useSelector } from 'react-redux';

import { YCSelectField } from '../../../../components/forms';
import { EXTERNAL_LINKS } from '../../../../models';
import { useRafDelay } from '../../../../utils';
import { useAddedSecretVersions } from '../../hooks';

import { SecretVersion } from '../../models';
import { loadSecretVersions, selectSecretVersions } from '../../slice';
import { RootStateWithSecrets } from '../../slice/state';

interface SecretVersionFieldControlProps {
   secretUuid?: string;

   /**
    * Автоматический выбор последней доступной версии
    */
   selectLatest: boolean;
}

/**
 * Компонент для выбора всех версий секрета. Работает через redux.
 */
export const SecretVersionField = React.memo(
   (props: ExtendedFieldConfig<any, string, SecretVersionFieldControlProps>) => {
      const { onChange, onBlur } = useExtendedField<string>(props);
      const dispatch = useDispatch();
      const delayedOnBlur = useRafDelay(onBlur);
      const { secretUuid, selectLatest } = props.controlProps ?? {};
      const addedSecretVersions = useAddedSecretVersions(secretUuid);

      const versionsSelector = useCallback(
         (s: RootStateWithSecrets) => (secretUuid ? selectSecretVersions(s, secretUuid) : []),
         [secretUuid],
      );
      const allSecretVersions = useSelector(versionsSelector);

      const versionToSelectItem = useCallback(
         (version: SecretVersion): YCSelectItem => ({
            disabled: addedSecretVersions?.some(v => v === version.version),
            key: version.version,
            title: version.createdAt ? formatDate(new Date(version.createdAt)) : version.version,
            url: secretUuid ? EXTERNAL_LINKS.secretVersion(secretUuid, version.version) : undefined,
            value: version.version,
         }),
         [addedSecretVersions, secretUuid],
      );

      const versionItems: YCSelectItem[] = useMemo(() => allSecretVersions?.map(versionToSelectItem) ?? [], [
         allSecretVersions,
         versionToSelectItem,
      ]);

      useEffect(() => {
         if (secretUuid) {
            dispatch(loadSecretVersions([secretUuid]));
         }
      }, [dispatch, secretUuid]);

      const shouldSelectLatest =
         selectLatest &&
         secretUuid &&
         !isEmpty(allSecretVersions) &&
         !addedSecretVersions?.includes(allSecretVersions[0].version);

      useEffect(() => {
         // Авто-выбор последней версии (спорное решение)
         if (shouldSelectLatest) {
            const v = allSecretVersions?.[0].version;
            if (v) {
               onChange(v);
               delayedOnBlur();
            }
         }
      }, [allSecretVersions, shouldSelectLatest, onChange, delayedOnBlur]);

      return (
         <YCSelectField<'single'>
            {...props}
            controlProps={{
               applyOnOutsideClick: false,
               type: 'single',
               items: versionItems,
               placeholder: 'Select version',
               showSearch: false,
            }}
         />
      );
   },
);

SecretVersionField.displayName = 'SecretVersionField';
