import { differenceInCalendarDays, lightFormat } from 'date-fns';
import { object, string } from 'yup';

import { LinkToSecret } from '../../models/ui/secrets';
import { getSequentialId } from '../../utils';
import { Secret, SecretVersion } from './models';

const generatedSecretAlias = /^sec-([0-9a-z]){26}:ver-([0-9a-z]){26}$/;

export function isSecretAliasGenerated(alias: string): boolean {
   return generatedSecretAlias.test(alias);
}

export function isSecretAliasGeneratedCorrectly(alias: string, secret: string, version: string): boolean {
   return alias === createSecretAlias(secret, version);
}

export function createSecretAlias(secret: string, version: string): string {
   return `${secret}:${version}`;
}

/**
 * Задаёт "сигнатуру" для токена
 *
 * В терминах секретницы это пароль для использования токена
 */
export function createSecretSignature(stageId: string, duId: string): string {
   return `${stageId}.${duId}`;
}

export interface SecretValidationContext {
   usedAliases: string[];
}

export const secretAliasSchema = string()
   .label('Alias')
   .required()
   .matches(/^[\w.:-]+$/) // Латинские буквы, знак подчеркивания, двоеточие, точка и тире. Никаких пробелов.
   .test({
      name: 'uniqueAlias',
      // eslint-disable-next-line no-template-curly-in-string
      message: '${path} "${value}" has already used in this deploy unit',
      test(v: string | undefined) {
         const { usedAliases }: SecretValidationContext = this.options.context as any;

         return v ? !usedAliases.includes(v) : true;
      },
   });

export const getLinkToSecretValidationSchema = (prefix = '') =>
   object<LinkToSecret>({
      key: string().min(1).label(`${prefix}Secret key`).required(),
      alias: string().min(1).label(`${prefix}Secret alias`).required(),
   });

/**
 * Возвращает список доступных для использования алиасов при заведении секрета/версии и при редактировании алиасов.
 *
 * Наиболее подходящий вариант идёт первым.
 *
 * TODO добавить несколько вариантов, сейчас только один(
 */
export function getAvailableAliases(
   secret: Secret,
   allSecretVersions: SecretVersion[] | undefined,
   usedAliases: string[],
   versionUuid?: string,
): string[] {
   const defaultAlias = secret.name ?? secret.uuid;

   // Алиас для конкретной версии
   if (versionUuid) {
      const version = allSecretVersions?.find(v => v.version === versionUuid);
      const versionDate = version?.createdAt ? new Date(version.createdAt) : null;

      const hasVersionsOnSameDay =
         allSecretVersions && version && versionDate
            ? allSecretVersions.some(
                 v =>
                    v.version !== version.version &&
                    v.createdAt &&
                    differenceInCalendarDays(new Date(v.createdAt), versionDate) === 0,
              )
            : false;

      const alias = createSecretAlias(
         defaultAlias,
         version && versionDate
            ? lightFormat(versionDate, hasVersionsOnSameDay ? 'yyyyMMddHHmm' : 'yyyyMMdd')
            : versionUuid,
      );

      return [getSequentialId(alias, usedAliases)];
   }

   // Есть только секрет
   return [getSequentialId(defaultAlias, usedAliases)];
}
