import { Dictionary } from '@reduxjs/toolkit';
import { useCallback, useMemo } from 'react';
import { useSelector } from 'react-redux';

import { createDictionaryReducer } from '../../utils';
import { useSecretsContext } from './hooks/useSecretsContext';

import { DeployUnitSecret, Secret, SecretVersion } from './models';
import { secretsSelector, selectAliasesFromDuSecrets, selectDeployUnitSecrets, selectSecretsVersions } from './slice';
import { RootStateWithSecrets } from './slice/state';

// TODO После окончания работ над модулем секретов удалить ненужные хуки (заинлайнить)
export function useDuSecrets(stageId: string, duId: string) {
   const selector = useCallback((s: RootStateWithSecrets) => selectDeployUnitSecrets(s, stageId, duId), [
      duId,
      stageId,
   ]);

   return useSelector(selector);
}

function useSecrets(secretUuids: string[]) {
   const selector = useCallback(
      (rootState: RootStateWithSecrets) =>
         secretUuids
            .map(secretUuid => secretsSelector.selectById(rootState, secretUuid))
            .reduce(
               createDictionaryReducer(s => s.uuid),
               {} as Dictionary<Secret>,
            ),
      [secretUuids],
   );

   return useSelector(selector);
}

function useSecretVersions(secretUuids: string[]) {
   const selector = useCallback((s: RootStateWithSecrets) => selectSecretsVersions(s, secretUuids), [secretUuids]);

   return useSelector(selector);
}

export interface DuSecretsWithVersions {
   duSecrets: DeployUnitSecret[] | undefined;
   secrets: Dictionary<Secret | undefined>;
   versions: Dictionary<SecretVersion[]>;
}

export function useDuSecretsWithVersions(stageId: string, duId: string): DuSecretsWithVersions {
   const duSecrets = useDuSecrets(stageId, duId);

   const secretUuids = useMemo(() => duSecrets?.map(x => x.secretUuid) ?? [], [duSecrets]);

   return {
      duSecrets,
      secrets: useSecrets(secretUuids),
      versions: useSecretVersions(secretUuids),
   };
}

/**
 * Возвращает список уже используемых в DU секретов (их uuid)
 *
 * Используется для предотвращения добавления дублей
 */
export function useAddedSecrets(): string[] {
   const { stageId, duId } = useSecretsContext();
   const { duSecrets } = useDuSecretsWithVersions(stageId, duId);

   return duSecrets?.map(s => s.secretUuid) ?? [];
}

/**
 * Возвращает список уже используемых в DU версий секрета (их uuid)
 *
 * Используется для предотвращения добавления дублей
 */
export function useAddedSecretVersions(secretUuid: string | undefined): string[] {
   const { stageId, duId } = useSecretsContext();
   const { duSecrets } = useDuSecretsWithVersions(stageId, duId);
   if (!secretUuid) {
      return [];
   }

   const duSecret = duSecrets?.find(s => s.secretUuid === secretUuid);

   return duSecret?.versions.map(v => v.versionUuid) ?? [];
}

/**
 * Возвращаем список существующих в DU алиасов секретов
 *
 * Используется для валидации уникальности
 */
export function useUsedAliases() {
   const { stageId, duId } = useSecretsContext();
   const { duSecrets } = useDuSecretsWithVersions(stageId, duId);

   return selectAliasesFromDuSecrets(duSecrets);
}
