import { isEmpty, useDismounted } from '@yandex-infracloud-ui/libs';
import { useCallback, useEffect, useMemo, useState } from 'react';
import { useSelector } from 'react-redux';
import { takeUntil } from 'rxjs/operators';

import { Stage } from '../../../models/ui';
import { TokenStore } from '../models';
import { getSecretsApi } from '../services';
import { RootStateWithSecrets, selectTokenRecords } from '../slice';

export function useActualSecretTokens(
   onlyEmpty: boolean,
   stage: Stage,
   duWithSecretsMigration: Set<string>, // игнорируем все секреты
): [TokenStore, boolean, number] {
   const dismounted = useDismounted();
   const [tokens, setTokens] = useState<TokenStore>([]);
   const [isLoading, setIsLoading] = useState(false);

   const tokenStoreSelector = useCallback((s: RootStateWithSecrets) => selectTokenRecords(s, stage), [stage]);
   const allSecrets: TokenStore = useSelector(tokenStoreSelector);

   const allOldSecrets = useMemo(() => allSecrets.filter(v => !duWithSecretsMigration.has(v.duId)), [
      allSecrets,
      duWithSecretsMigration,
   ]);

   const reloadTokens = useCallback(
      (store: TokenStore) => {
         setIsLoading(true);
         getSecretsApi()
            .getDelegationTokens(store)
            .pipe(takeUntil(dismounted))
            .subscribe(actualTokens => {
               setIsLoading(false);
               setTokens(actualTokens);
            });
      },
      [dismounted],
   );

   useEffect(() => {
      const store = allOldSecrets.filter(secret => (onlyEmpty ? isEmpty(secret.token) : true));

      reloadTokens(store);
   }, [allOldSecrets, onlyEmpty, reloadTokens]);

   return [tokens, isLoading, allOldSecrets.length];
}
