import {
   ApiCache,
   BaseApi,
   HttpMethod,
   HttpStatusCode,
   IApiError,
   isEmpty,
   sortHandler,
} from '@yandex-infracloud-ui/libs';
import { forkJoin, Observable, of } from 'rxjs';
import { catchError, map } from 'rxjs/operators';

import { noop } from '../../../utils';
import { createSecretSignature } from '../helpers';

import { Secret, SecretApiModel, SecretVersion, SecretVersionApiModel, TokenStore } from '../models';

interface GetListInput {
   query: string;
}

interface GetListPayload {
   secrets: SecretApiModel[];
}

interface GetVersionsInput {
   secretUuid: string;
}

interface GetVersionsPayload {
   versions: SecretVersionApiModel[];
}

interface CreateDelegationTokenInput {
   secretUuid: string;
   signature: string;
}

interface CreateDelegationTokenPayload {
   token: { token: string; uuid: string };
}

export class SecretsApi extends BaseApi {
   private cache = new ApiCache();

   constructor(deployHost: string) {
      super(`${deployHost}/api/yav/`);
   }

   public getById(secretUuid: string): Observable<Secret> {
      return this.getList(secretUuid).pipe(
         map(secrets => {
            const deniedSecret = { denied: true, uuid: secretUuid };

            return secrets.find(s => s.uuid === secretUuid) ?? deniedSecret;
         }),
      );
   }

   public getList(query: string): Observable<Secret[]> {
      if (isEmpty(query)) {
         return of([]);
      }

      const params: GetListInput = { query };

      function toSecret(s: SecretApiModel): Secret {
         return {
            createBy: s.createBy,
            createdAt: new Date(s.createdAt).getTime(),
            denied: false,
            name: s.name,
            uuid: s.uuid,
         };
      }

      return this.cache.request(`getList_${query}`, () =>
         this.request<GetListInput, void, GetListPayload>(HttpMethod.GET, 'ListSecrets/', params).pipe(
            map(resp => resp.secrets.map(toSecret)),
         ),
      );
   }

   public getVersions(secretUuid: string): Observable<SecretVersion[]> {
      const params = { secretUuid };

      return this.cache.request(`getVersions_${secretUuid}`, () =>
         this.request<GetVersionsInput, void, GetVersionsPayload>(HttpMethod.GET, 'ListSecretVersions/', params).pipe(
            map(resp =>
               resp.versions
                  .map(
                     v =>
                        ({
                           version: v.version,
                           createdBy: v.createdBy,
                           keys: v.keys,
                           createdAt: new Date(v.createdAt).getTime(),
                        } as SecretVersion),
                  )
                  .sort((a, b) => sortHandler(b.createdAt, a.createdAt)),
            ),
            catchError(err => {
               if (err.code === HttpStatusCode.Forbidden) {
                  return of([]);
               }

               throw err;
            }),
         ),
      );
   }

   public createDelegationToken(secretUuid: string, signature: string): Observable<string> {
      const params: CreateDelegationTokenInput = { secretUuid, signature };

      return this.cache.request(`createDelegationToken_${params.secretUuid}_${params.signature}`, () =>
         this.request<CreateDelegationTokenInput, void, CreateDelegationTokenPayload>(
            HttpMethod.GET,
            'CreateDelegationToken/',
            params,
         ).pipe(map(resp => resp.token.token)),
      );
   }

   public getDelegationTokens(secrets: TokenStore): Observable<TokenStore> {
      if (secrets.length === 0) {
         return of([]);
      }

      const observables = secrets.map(({ stageId, duId, secretUuid }) => {
         const signature = createSecretSignature(stageId, duId);

         // Ошибки пока игнорируем
         // (даже если для какого-то секрета не удалось выписать токен, целиком запрос не должен упасть)
         return this.createDelegationToken(secretUuid, signature).pipe(catchError(() => of(undefined)));
      });

      return forkJoin(observables).pipe(
         map(tokens =>
            tokens.map((token, i) => ({
               ...secrets[i],
               token,
            })),
         ),
      );
   }

   protected getCSRF(): string {
      return '';
   }

   protected handleError(resp: Response, error: IApiError): void {
      noop();
   }
}
