import { createSelector, Dictionary } from '@reduxjs/toolkit';
import { isEmpty, unique } from '@yandex-infracloud-ui/libs';
import { Stage } from '../../../models/ui';

import { createDictionary, notEmpty } from '../../../utils';

import { DeployUnitSecret, SecretVersion, TokenRecord, TokenStore } from '../models';
import { namespace, RootStateWithSecrets, secretAdapter, versionAdapter } from './state';

export const secretsSelector = secretAdapter.getSelectors<RootStateWithSecrets>(s => s[namespace].secrets);

const versionSelector = versionAdapter.getSelectors<RootStateWithSecrets>(s => s[namespace].versions);

type SelectDeployUnitSecretsSignature = (
   state: RootStateWithSecrets,
   stageId: string,
   duId: string,
) => DeployUnitSecret[] | undefined;

export const selectStageSecrets = createSelector(
   (state: RootStateWithSecrets) => state[namespace].deployUnitSecrets,
   (state: RootStateWithSecrets, stageId: string) => stageId,

   (state, stageId) => state[stageId],
);

export const selectDeployUnitSecrets: SelectDeployUnitSecretsSignature = createSelector(
   (state: RootStateWithSecrets) => state[namespace].deployUnitSecrets,
   (state: RootStateWithSecrets, stageId: string) => stageId,
   (state: RootStateWithSecrets, stageId: string, duId: string) => duId,

   (state, stageId, duId) => state[stageId]?.[duId],
);

/**
 * Бежим по всем секретам, загруженных из стейджей,
 * чтобы собрать информацию о версиях, которые недоступны из API (нет доступа у пользователя)
 * Так можно хоть как-то работать с denied секретами (выбирать их и их ключи)
 */
const selectSecretVersionsFromDu = createSelector(
   (state: RootStateWithSecrets) => state[namespace].deployUnitSecrets,
   (state: RootStateWithSecrets, secretUuid: string) => secretUuid,
   (duState, secretUuid) => {
      const fromDuVersions: SecretVersion[] = [];

      for (const secretsDict of Object.values(duState)) {
         for (const duSecrets of Object.values(secretsDict)) {
            for (const duSecret of duSecrets) {
               if (duSecret.secretUuid === secretUuid) {
                  for (const version of duSecret.versions) {
                     const exist = fromDuVersions.find(v => v.version === version.versionUuid);

                     // key === '' - временный костыль для мультисекретов
                     if (exist) {
                        exist.keys = unique([...exist.keys, ...version.usages.map(u => u.key)]).filter(v => v !== '');
                     } else {
                        fromDuVersions.push({
                           version: version.versionUuid,
                           keys: unique(version.usages.map(u => u.key)).filter(v => v !== ''),
                        });
                     }
                  }
               }
            }
         }
      }

      return fromDuVersions;
   },
);

type SelectSecretVersions = (state: RootStateWithSecrets, secretUuid: string) => SecretVersion[] | undefined;

export const selectSecretVersions: SelectSecretVersions = createSelector(
   (state: RootStateWithSecrets) => state, // TODO avoid passing rootState to selector!
   (state: RootStateWithSecrets) => state[namespace].versionsOfSecret,
   (state: RootStateWithSecrets, secretUuid: string) => secretUuid,

   (rootState, versionsState, secretUuid) => {
      const normalVersions =
         versionsState[secretUuid]?.map(v => versionSelector.selectById(rootState, v)!).filter(notEmpty) ?? [];
      const knownVersionUuids = new Set(normalVersions.map(nv => nv.version));

      const fromDuVersions = selectSecretVersionsFromDu(rootState, secretUuid).filter(
         duV => !knownVersionUuids.has(duV.version),
      );

      return [...normalVersions, ...fromDuVersions];
   },
);

type SelectSecretsVersionsSignature = (
   state: RootStateWithSecrets,
   secretUuids: string[],
) => Dictionary<SecretVersion[]>;

export const selectSecretsVersions: SelectSecretsVersionsSignature = createSelector(
   (state: RootStateWithSecrets) => state, // TODO avoid passing rootState to selector!
   (state: RootStateWithSecrets, secretUuids: string[]) => secretUuids,

   (rootState, secretUuids) =>
      secretUuids.reduce((acc, secretUuid) => {
         const secretVersions = selectSecretVersions(rootState, secretUuid);

         if (secretVersions && secretVersions.length > 0) {
            acc[secretUuid] = secretVersions;
         }

         return acc;
      }, {} as Dictionary<SecretVersion[]>),
);

export const selectSecret = secretsSelector.selectById;

export const selectSecrets = secretsSelector.selectEntities;

export const selectDuSecretsBundle = (state: RootStateWithSecrets, stageId: string, duId: string) => {
   const duSecrets = selectDeployUnitSecrets(state, stageId, duId);

   const secretUuids = duSecrets?.map(x => x.secretUuid) ?? [];

   const secretList = secretUuids.map(secretUuid => secretsSelector.selectById(state, secretUuid));

   return {
      duSecrets,
      secrets: createDictionary(secretList, s => s.uuid),
      versions: selectSecretsVersions(state, secretUuids),
   };
};

/**
 * Ищет информацию о секрете по его использованию (т.е. по алиасу)
 *
 * Алиас является связкой между местом использования секрета и его хранилищем в рамках DU.
 *
 * Пока что используется только для компонента SelectedSecretName в переменных окружения.
 */
export const selectDuSecretByAlias = (
   state: RootStateWithSecrets,
   stageId: string,
   duId: string,
   alias: string,
): DeployUnitSecret | undefined => {
   const duSecrets = selectDeployUnitSecrets(state, stageId, duId);

   return duSecrets?.find(s => s.versions.some(v => v.alias === alias));
};

/**
 * Ищет подходящий токен в других версиях того же секрета при добавлении новой версии
 *
 * Так как токен не зависит от KEY или версии, их можно переиспользовать от того же секрета в том же DU
 */
export const selectDelegationToken = (
   state: RootStateWithSecrets,
   stageId: string,
   duId: string,
   secretUuid: string,
): string | undefined => {
   const duSecrets = selectDeployUnitSecrets(state, stageId, duId);

   const duVersions = duSecrets?.find(s => s.secretUuid === secretUuid)?.versions;

   return duVersions?.find(v => !isEmpty(v.token))?.token;
};

type SelectTokenRecordsSignature = (state: RootStateWithSecrets, stage: Stage) => TokenStore;

export const selectTokenRecords: SelectTokenRecordsSignature = createSelector(
   (state: RootStateWithSecrets) => state[namespace].deployUnitSecrets,
   (state: RootStateWithSecrets, stage: Stage) => stage,
   (duSecretsState, stage) => {
      const stageSecrets = duSecretsState[stage.initialId ?? stage.id];

      return Object.entries(stageSecrets ?? {})
         .map(([initialDuId, duSecrets]) =>
            duSecrets
               // Оставляем секреты, если у него есть неудаленные версии
               .filter(duSecret => duSecret.versions.some(v => !v.removed))
               // Оставляем секреты, если есть версия в старом хранилище (для нового токены не нужны)
               .filter(duSecret => duSecret.versions.some(v => v.legacy))
               .map(
                  duSecret =>
                     ({
                        stageId: stage.id,
                        duId: resolveActualDuId(stage, initialDuId),
                        secretUuid: duSecret.secretUuid,
                        // Первый непустой токен из секретов
                        token: duSecret.versions.find(v => !isEmpty(v.token))?.token,
                     } as TokenRecord),
               ),
         )
         .flat();
   },
);

function resolveActualDuId(stage: Stage, initialDuId: string): string {
   const existDu = stage.deployUnits.find(du => du.initialId === initialDuId);

   return existDu ? existDu.id : initialDuId;
}

export const selectAliasesFromDuSecrets = (
   duSecrets: DeployUnitSecret[] | undefined,
   ignoreAliases?: (string | undefined)[],
): string[] =>
   duSecrets
      ? unique(
           duSecrets.reduce((acc, duSecret) => {
              for (const duVersion of duSecret.versions) {
                 if (duVersion.alias && !ignoreAliases?.includes(duVersion.alias)) {
                    acc.push(duVersion.alias);
                 }
                 if (duVersion.newAlias && !ignoreAliases?.includes(duVersion.newAlias)) {
                    acc.push(duVersion.newAlias);
                 }
              }

              return acc;
           }, [] as string[]),
        )
      : [];
