// noinspection SpellCheckingInspection

import { configureStore, EnhancedStore } from '@reduxjs/toolkit';
import thunk from 'redux-thunk';

import { StageConverter } from '../../../models/ui';
import { testSecretsStage } from '../../../stories/testData';
import { SECRETS_DEMO_STAGE_ID } from '../storybook/StageWithSecretStory';
import { selectSecretsVersions } from './selectors';
import { secretsSlice } from './slice';
import { namespace, RootStateWithSecrets, SecretStore } from './state';

describe('module/secrets/slice', () => {
   let store: EnhancedStore<RootStateWithSecrets>;

   beforeEach(() => {
      store = configureStore({
         middleware: [thunk],
         reducer: {
            [namespace]: secretsSlice.reducer,
         },
      });
   });

   function loadStage(): RootStateWithSecrets {
      store.dispatch(
         secretsSlice.actions.loadFromStage({
            rawStageSpec: testSecretsStage.spec as any,
            stage: StageConverter.fromApi(testSecretsStage),
            recreate: true,
         }),
      );

      return store.getState();
   }

   it('loadFromStage', () => {
      const state: SecretStore = loadStage()[namespace];
      const duDict = state.deployUnitSecrets[SECRETS_DEMO_STAGE_ID];

      expect(duDict.du_manual_secrets).toMatchInlineSnapshot(`
         Array [
           Object {
             "secretUuid": "sec-01dec420c9rpdgdc9p3astjs0x",
             "versions": Array [
               Object {
                 "alias": "INFRACLOUDUI_new",
                 "usages": Array [
                   Object {
                     "key": "ARC_TOKEN",
                     "path": "boxes[box].environment[ARC_TOKEN]",
                     "type": "environment",
                   },
                 ],
                 "versionUuid": "ver-01erj3s88hsyahp063efjsd12m",
               },
               Object {
                 "alias": "INFRACLOUDUI",
                 "legacy": true,
                 "token": "kdVLi5hao80j_mXlJcCCoJY3Pir27Rvg0n8WX8J3d04.1.dcd524be52af6b63",
                 "usages": Array [
                   Object {
                     "key": "YT_TOKEN",
                     "path": "boxes[box].environment[YT_TOKEN]",
                     "type": "environment",
                   },
                   Object {
                     "key": "FONTAWESOME_NPM_AUTH_TOKEN",
                     "path": "boxes[box].environment[FONTAWESOME_NPM_AUTH_TOKEN]",
                     "type": "environment",
                   },
                   Object {
                     "key": "YT_TOKEN",
                     "path": "boxes[box].workloads[workload].environment[YT_TOKEN]",
                     "type": "environment",
                   },
                 ],
                 "versionUuid": "ver-01erj3s88hsyahp063efjsd12m",
               },
             ],
           },
           Object {
             "secretUuid": "sec-01d6bkr38deswha5zrszemkm50",
             "versions": Array [
               Object {
                 "alias": "YET_ANOTHER_DENIED_SECRET_ALIAS",
                 "legacy": true,
                 "token": "xTJ6-Mg-ZaxOCU-3X0FnMVsQEO_BKvFrJVW6cMPxHAA.1.b5f441e03a5c2095",
                 "usages": Array [],
                 "versionUuid": "ver-01dvzmdb0pxdg9z1t60few7s70",
               },
               Object {
                 "alias": "DENIED_SECRET",
                 "legacy": true,
                 "token": "xTJ6-Mg-ZaxOCU-3X0FnMVsQEO_BKvFrJVW6cMPxHAA.1.b5f441e03a5c2095",
                 "usages": Array [
                   Object {
                     "key": "reddi-test",
                     "path": "boxes[box].workloads[workload].environment[DENIED_VAR]",
                     "type": "environment",
                   },
                 ],
                 "versionUuid": "ver-01dvzmdb0pxdg9z1t60few7s70",
               },
             ],
           },
         ]
      `);
   });

   const DENIED_SECRET_UUID = 'sec-01d6bkr38deswha5zrszemkm50';

   it('select keys for denied secrets', () => {
      const state = loadStage();

      expect(selectSecretsVersions(state, [DENIED_SECRET_UUID])).toMatchInlineSnapshot(`
         Object {
           "sec-01d6bkr38deswha5zrszemkm50": Array [
             Object {
               "keys": Array [
                 "reddi-test",
               ],
               "version": "ver-01dvzmdb0pxdg9z1t60few7s70",
             },
           ],
         }
      `);
   });
});
