import { createSlice, PayloadAction } from '@reduxjs/toolkit';
import { getSetDifference, isEmpty, omitFields, sortHandler } from '@yandex-infracloud-ui/libs';

import { DeployUnitConverter, Stage } from '../../../models/ui';
import { patchList, patchObject } from '../../../models/ui/utils';
import { TStageSpec } from '../../../proto-typings';

import { DeployUnitSecret, DeployUnitSecretVersion, Secret } from '../models';
import { loadSecret } from './asyncActions/loadSecret';
import { loadSecrets, LoadSecretsOutput } from './asyncActions/loadSecrets';
import { loadSecretVersions, LoadSecretVersionsOutput } from './asyncActions/loadSecretVersions';
import {
   upsertDeployUnitSecretVersion,
   UpsertDeployUnitSecretVersionParams,
} from './asyncActions/upsertDeployUnitSecretVersion';
import { initialState, namespace, secretAdapter, SecretStore, versionAdapter } from './state';

interface RemoveUnusedVersionsFromSecretPayload {
   duId: string;
   secretUuid: string;
   stageId: string;
}

interface LoadFromStagePayload {
   rawStageSpec: TStageSpec;
   stage: Stage;
   recreate: boolean;
}

interface CloneDeployUnitPayload {
   stageId: string;
   originalInitialId: string;
   newDuId: string;
}

interface MigrateFromLegacyPayload {
   stageId: string;
   duId: string;
   alias?: string;
}

interface EditAliasPayload {
   stageId: string;
   duId: string;
   duVersion: DeployUnitSecretVersion;
   newAlias: string;
}

export const secretsSlice = createSlice({
   name: namespace,
   initialState,
   reducers: {
      addSecrets(state, action: PayloadAction<Secret[]>) {
         secretAdapter.addMany(state.secrets, action.payload);
      },
      loadFromStage(state, action: PayloadAction<LoadFromStagePayload>) {
         const { stage, rawStageSpec, recreate } = action.payload;
         const stageId = stage.initialId ?? stage.id;

         // Заполняем заново
         if (recreate) {
            delete state.deployUnitSecrets[stageId];
         }

         for (const du of stage.deployUnits) {
            const duId = du.initialId ?? du.id;
            const duSpec = rawStageSpec.deploy_units?.[duId];

            // Из спеки DU извлекаются все секреты
            if (duSpec) {
               const podTemplateSpec = DeployUnitConverter.getPodTemplateSpec(duSpec)!;

               // actual storage
               const podSpecSecrets = podTemplateSpec.spec?.secret_refs ?? {};
               patchDeploySecrets(state, stageId, duId, secrets => {
                  Object.entries(podSpecSecrets).forEach(([alias, record]) => {
                     const existSecret = secrets.find(s => s.secretUuid === record.secret_id);

                     const newVersion: DeployUnitSecretVersion = {
                        alias,
                        usages: [], // Будут заполнены чуть позже (в этом же action перед выходом из for)
                        versionUuid: record.secret_version,
                     };

                     if (existSecret) {
                        const existVersion = existSecret.versions.find(v => v.alias === newVersion.alias);
                        if (!existVersion) {
                           existSecret.versions.push(newVersion);
                        } else {
                           Object.assign(existVersion, newVersion);
                        }
                     } else {
                        secrets.push({
                           secretUuid: record.secret_id,
                           versions: [newVersion],
                        });
                     }
                  });

                  return secrets;
               });

               // legacy storage
               const podSpecSecretsLegacy = podTemplateSpec.spec?.secrets ?? {};
               patchDeploySecrets(state, stageId, duId, secrets => {
                  Object.entries(podSpecSecretsLegacy).forEach(([alias, record]) => {
                     const existSecret = secrets.find(s => s.secretUuid === record.secret_id);

                     const newVersion: DeployUnitSecretVersion = {
                        alias,
                        legacy: true,
                        token: record.delegation_token ?? undefined,
                        usages: [], // Будут заполнены чуть позже (в этом же action перед выходом из for)
                        versionUuid: record.secret_version,
                     };

                     if (existSecret) {
                        const existVersion = existSecret.versions.find(v => v.alias === newVersion.alias);
                        if (!existVersion) {
                           existSecret.versions.push(newVersion);
                        } else {
                           Object.assign(existVersion, newVersion);
                        }
                     } else {
                        secrets.push({
                           secretUuid: record.secret_id,
                           versions: [newVersion],
                        });
                     }
                  });

                  return secrets;
               });
            }

            // Обновление использования секретов для всего: новых и старых DU, новых и старых секретов
            patchDeploySecrets(state, stageId, duId, secrets => {
               // Обновляем известные секреты
               for (const secret of secrets) {
                  for (const version of secret.versions) {
                     version.usages = DeployUnitConverter.getSecretUsages(du, version.alias);
                  }
               }

               return secrets;
            });
         }
      },
      cloneDeployUnit(state, action: PayloadAction<CloneDeployUnitPayload>) {
         const { stageId, originalInitialId, newDuId } = action.payload;

         const oldSecrets = state.deployUnitSecrets[stageId]?.[originalInitialId] ?? [];

         patchDeploySecrets(state, stageId, newDuId, secrets => {
            for (const secret of oldSecrets) {
               secrets.push({
                  ...secret,
                  // токен надо затирать, т.к. он зависит от duId
                  versions: secret.versions.map(sv => ({ ...sv, token: undefined })),
               });
            }

            return secrets;
         });
      },
      removeUnusedVersionsFromSecret(state, action: PayloadAction<RemoveUnusedVersionsFromSecretPayload>) {
         const { stageId, duId, secretUuid } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            for (const secret of secrets) {
               if (secret.secretUuid === secretUuid) {
                  for (const version of secret.versions) {
                     if (version.usages.length === 0) {
                        version.removed = true;
                     }
                  }
               }
            }

            return secrets;
         });
      },
      migrateFromLegacy(state, action: PayloadAction<MigrateFromLegacyPayload>) {
         const { stageId, duId, alias } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            for (const duSecret of secrets) {
               const oldDuVersion = duSecret.versions.find(v => v.alias === alias && v.legacy);
               if (oldDuVersion) {
                  migrateVersion(oldDuVersion, duSecret);
               }
            }

            return secrets;
         });
      },
      migrateAllFromLegacy(state, action: PayloadAction<MigrateFromLegacyPayload>) {
         const { stageId, duId } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            for (const duSecret of secrets) {
               const oldVersions = duSecret.versions.filter(v => v.legacy && !v.migrated);
               for (const oldDuVersion of oldVersions) {
                  migrateVersion(oldDuVersion, duSecret);
               }
            }
            return secrets;
         });
      },
      undoMigrationFromLegacy(state, action: PayloadAction<MigrateFromLegacyPayload>) {
         const { stageId, duId, alias } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            for (const duSecret of secrets) {
               const newDuVersion = duSecret.versions.find(v => v.alias === alias && !v.legacy && !v.migrated);
               const oldDuVersion = duSecret.versions.find(v => v.alias === alias && v.legacy && v.migrated);
               if (newDuVersion && oldDuVersion) {
                  // remove new version
                  duSecret.versions = duSecret.versions.filter(v => v !== newDuVersion);

                  // delete removing mark for old version
                  delete oldDuVersion.removed;
                  delete oldDuVersion.migrated;
               }
            }

            return secrets;
         });
      },
      editAlias(state, action: PayloadAction<EditAliasPayload>) {
         const { stageId, duId, duVersion, newAlias } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            for (const duSecret of secrets) {
               const duVersionFromState = duSecret.versions.find(v => v.alias === duVersion.alias);
               if (duVersionFromState) {
                  // Отмена редактирования
                  if (duVersionFromState.alias === newAlias && duVersionFromState.newAlias) {
                     delete duVersionFromState.newAlias;
                  }

                  // Установка нового значения
                  if (duVersionFromState.alias !== newAlias) {
                     duVersionFromState.newAlias = newAlias;
                  }
               }
            }
            return secrets;
         });
      },
   },
   extraReducers: {
      [loadSecret.fulfilled.toString()](state, action: PayloadAction<Secret>) {
         secretAdapter.upsertOne(state.secrets, action.payload);
      },
      [loadSecrets.fulfilled.toString()](state, action: PayloadAction<LoadSecretsOutput[]>) {
         for (const { secret, versions } of action.payload) {
            secretAdapter.upsertOne(state.secrets, secret);
            versionAdapter.upsertMany(state.versions, versions);

            addVersionsToSecret(
               state,
               secret.uuid,
               versions.map(v => v.version),
            );
         }
      },
      [loadSecretVersions.fulfilled.toString()](state, action: PayloadAction<LoadSecretVersionsOutput>) {
         const data = action.payload;

         for (const secretUuid in data) {
            if (data.hasOwnProperty(secretUuid)) {
               const versions = data[secretUuid];

               // Если смогли загрузить версии, значит доступ есть, иначе нет
               secretAdapter.updateOne(state.secrets, { id: secretUuid, changes: { denied: isEmpty(versions) } });

               versionAdapter.upsertMany(state.versions, versions);

               addVersionsToSecret(
                  state,
                  secretUuid,
                  versions.map(v => v.version),
               );
            }
         }
      },
      [upsertDeployUnitSecretVersion.fulfilled.toString()](
         state,
         action: PayloadAction<UpsertDeployUnitSecretVersionParams>,
      ) {
         const { stageId, duId, secretUuid, versionUuid, alias, token, newStorage } = action.payload;

         patchDeploySecrets(state, stageId, duId, secrets => {
            const existSecret = secrets.find(s => s.secretUuid === secretUuid);

            const newVersion: DeployUnitSecretVersion = {
               alias,
               token,
               usages: [],
               versionUuid,
            };

            if (!newStorage) {
               newVersion.legacy = true;
            }

            if (existSecret) {
               const existVersion = existSecret.versions.find(v => v.versionUuid === versionUuid);

               if (existVersion) {
                  existVersion.alias = alias;
                  existVersion.token = token;
               } else {
                  existSecret.versions.push(newVersion);

                  // Обратный хронологический порядок (versionUuid убывает)
                  existSecret.versions.sort((a, b) => sortHandler(b.versionUuid, a.versionUuid));
               }
            } else {
               secrets.push({
                  secretUuid,
                  versions: [newVersion],
               });
            }

            return secrets;
         });

         addVersionsToSecret(state, secretUuid, [versionUuid]);
      },
   },
});

/**
 * Патчит секреты в state.deployUnitSecrets
 */
function patchDeploySecrets(
   state: SecretStore,
   stageId: string,
   duId: string,
   patcher: (secrets: DeployUnitSecret[]) => DeployUnitSecret[] | undefined,
) {
   patchObject(state.deployUnitSecrets, stageId, deployUnits => {
      patchList(deployUnits, duId, patcher);

      return deployUnits;
   });
}

/**
 * Добавляет известные версии к секрету
 */
function addVersionsToSecret(state: SecretStore, secretUuid: string, versions: string[]) {
   patchList(state.versionsOfSecret, secretUuid, secretVersionUuids => {
      const newVersionUuids = new Set(versions);
      const { added } = getSetDifference(new Set(secretVersionUuids), newVersionUuids);

      added.forEach(newVersionUuid => secretVersionUuids.push(newVersionUuid));

      return secretVersionUuids;
   });
}

function migrateVersion(oldDuVersion: DeployUnitSecretVersion, duSecret: DeployUnitSecret) {
   const newDuVersion = omitFields(oldDuVersion, 'legacy', 'token') as DeployUnitSecretVersion;
   duSecret.versions.push(newDuVersion);

   oldDuVersion.removed = true;
   oldDuVersion.migrated = true;
}
