import { ExternalLink } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { firstValueFrom } from 'rxjs';
import { EXTERNAL_LINKS } from '../../../models';
import { ypApi } from '../../../services';
import { capacityConfigurations } from '../../store/reducers/balancers';
import BalancersFormSection from '../BalancersFormSection/BalancersFormSection';
import CheckBox from '../CheckBox/CheckBox';
import Select from '../Select/Select';
import Suggest from '../Suggest/Suggest';
import TextInput from '../TextInput/TextInput';

import './BalancersFormBalancerInstances.scss';

const b = block('balancer-instances');

export default class BalancersFormBalancerInstances extends Component {
   state = {
      clusters: this.props.clusters.map(x => ({ key: x, isChecked: this.props.stageClusters.includes(x) })),
      instancesPerCluster: 1,
      selectedCapacityConfig: capacityConfigurations[0].val,
      selectedMacro: '_SEARCHSAND_',
      macroSuggestItems: [],
      isClustersValid: true,
      isMacroValid: true,
   };

   macrosItems = [];

   componentDidMount() {
      const { login } = window.USER;
      firstValueFrom(ypApi.getAvailableNetworkProjects(login))
         .catch(() => [])
         .then(macrosItems => {
            this.macrosItems = macrosItems;
         });
   }

   componentDidUpdate(prevProps) {
      const { clusters, stageClusters } = this.props;
      const { clusters: prevClusters, stageClusters: prevStageClusters } = prevProps;
      if (clusters.length !== prevClusters.length || stageClusters.length !== prevStageClusters.length) {
         this.setState({
            clusters: clusters.map(x => ({ key: x, isChecked: stageClusters.includes(x) })),
         });
      }
   }

   onClusterChange(key, isChecked) {
      const { clusters } = this.state;
      const cluster = clusters.find(x => x.key === key);
      cluster.isChecked = isChecked;

      this.setState({ clusters: [...clusters] });
   }

   onNumberOfInstancesChange(newValue) {
      if (newValue >= 1) {
         this.setState({ instancesPerCluster: newValue });
      }
   }

   onCapacityConfigSelected([newValue]) {
      this.setState({ selectedCapacityConfig: newValue });
   }

   onMacroTextChanged(newValue) {
      const items = this.macrosItems.filter(macros => macros.includes(newValue.toUpperCase()));

      this.setState({ macroSuggestItems: items, selectedMacro: newValue });
   }

   onMacroSelected(item) {
      this.setState({ selectedMacro: item });
   }

   serialize() {
      const { clusters, selectedMacro, instancesPerCluster, selectedCapacityConfig } = this.state;

      return {
         locations: clusters.filter(x => x.isChecked).map(x => x.key.toUpperCase()),
         nannyServiceIdSlug: 'not-used-yet',
         networkMacro: selectedMacro,
         preset: {
            'instances_count': instancesPerCluster,
            'type': selectedCapacityConfig.toUpperCase(),
         },
         type: 'PRESET',
      };
   }

   validate() {
      const { clusters, selectedMacro } = this.state;
      const isClustersValid = clusters.some(x => x.isChecked);

      const isMacroValid = this.macrosItems.includes(selectedMacro);

      this.setState({ isClustersValid, isMacroValid });

      return isClustersValid && isMacroValid;
   }

   render() {
      const {
         clusters,
         instancesPerCluster,
         selectedCapacityConfig,
         selectedMacro,
         macroSuggestItems,
         isClustersValid,
         isMacroValid,
      } = this.state;
      const clusterCheckboxes = clusters.map(({ key, isChecked }) => (
         <CheckBox
            key={`cluster-${key}`}
            checked={isChecked}
            onChange={e => {
               this.onClusterChange(key, e.target.checked);
            }}
            cls={b('cluster')}
         >
            {key.toUpperCase()}
         </CheckBox>
      ));

      return (
         <BalancersFormSection title={'Balancer instances'} className={b()}>
            <p>Balancer instances will be created and configured according to these settings:</p>
            <div className={b('clusters-container')}>
               <div className={b('clusters-title')}>Clusters</div>
               <div className={b('clusters')}>{clusterCheckboxes}</div>
               <div className={b('error', { hidden: isClustersValid })}>At least one cluster must be selected</div>
            </div>
            <div className={b('capacity-container')}>
               <div className={b('capacity-title')}>
                  <span>Instances in each cluster</span>
                  <i className={`far fa-times ${b('cross')}`} />
                  <span>Capacity of each instance</span>
               </div>
               <div className={b('capacity-controls-container')}>
                  <TextInput
                     type={'number'}
                     text={instancesPerCluster}
                     cls={b('capacity-control')}
                     onChange={v => this.onNumberOfInstancesChange(v)}
                  />
                  <i className={`far fa-times ${b('cross')}`} />
                  <Select
                     items={capacityConfigurations}
                     val={selectedCapacityConfig}
                     onChange={v => this.onCapacityConfigSelected(v)}
                  />
               </div>
            </div>
            <div className={b('network-container')}>
               <div className={b('network-title')}>Network macro</div>
               <div className={b('network-control-container')}>
                  <div className={b('suggest')}>
                     <Suggest
                        data={macroSuggestItems}
                        value={selectedMacro}
                        renderItem={item => item}
                        onChange={v => this.onMacroTextChanged(v)}
                        onAction={item => this.onMacroSelected(item)}
                     />
                  </div>
                  <ExternalLink href={EXTERNAL_LINKS.wiki.createNetworkMacro}>Create new macro</ExternalLink>
               </div>
               <div className={b('hint')}>
                  Network macro affects clients → balancer and balancer → backends accessibility
               </div>
               <div className={b('error', { hidden: isMacroValid, macro: true })}>Invalid macro</div>
            </div>
         </BalancersFormSection>
      );
   }
}

BalancersFormBalancerInstances.propTypes = {
   stageClusters: PropTypes.arrayOf(PropTypes.string).isRequired,
   // eslint-disable-next-line react/forbid-prop-types
   clusters: PropTypes.array.isRequired,
};
