import block from 'bem-cn-lite';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import BalancersFormEndpointSet from '../BalancersFormEndpointSet/BalancersFormEndpointSet';
import BalancersFormSection from '../BalancersFormSection/BalancersFormSection';

import './BalancersFormEndpointSets.scss';

const b = block('endpoint-sets');

const mapStageEndpointSets = ({ cluster, id }) => ({
   cluster,
   name: id,
   isValid: true,
   id: `${cluster}:${id}`,
});

export default class BalancersFormEndpointSets extends Component {
   propTypes = {
      stageDeployUnits: PropTypes.array,
      clusters: PropTypes.array,
   };

   state = {
      endpointSets: this.props.stageDeployUnits.map(mapStageEndpointSets),
      validationMessage: '',
   };

   componentDidUpdate(prevProps) {
      const { stageDeployUnits } = this.props;
      if (prevProps.stageDeployUnits.length !== stageDeployUnits.length) {
         this.setState({
            endpointSets: stageDeployUnits.map(mapStageEndpointSets),
         });
      }
   }

   onClusterChange(id, newValue) {
      const { endpointSets } = this.state;
      const changedSet = endpointSets.find(x => x.id === id);
      changedSet.cluster = newValue;
      this.setState({ endpointSets: [...endpointSets] });
   }

   onEndpointSetNameChange(id, newValue) {
      const { endpointSets } = this.state;
      const changedSet = endpointSets.find(x => x.id === id);
      changedSet.name = newValue;
      this.setState({ endpointSets: [...endpointSets] });
   }

   onEndpointSetRemove(id) {
      const { endpointSets } = this.state;
      const setIndex = endpointSets.findIndex(x => x.id === id);
      endpointSets.splice(setIndex, 1);

      this.setState({ endpointSets: [...endpointSets] });
   }

   addEndpointSet() {
      const { clusters } = this.props;
      const { endpointSets } = this.state;
      endpointSets.push({ cluster: clusters[0], name: '', isValid: true, id: Date.now() });

      this.setState({ endpointSets: [...endpointSets] });
   }

   serialize() {
      const { endpointSets } = this.state;

      return endpointSets.map(({ name, cluster }) => ({ cluster, id: name }));
   }

   validate() {
      const { endpointSets } = this.state;
      const regex = /^[A-Za-z0-9-_.]+$/;
      endpointSets.forEach(set => {
         set.isValid = regex.test(set.name);
      });
      const validationResult = endpointSets
         .filter(x => x.isValid)
         .reduce(
            (result, set) => {
               const { data } = result;
               const key = `${set.cluster}_${set.name}`;
               if (data[key]) {
                  result.isValid = false;
                  data[key] += 1;
               } else {
                  data[key] = 1;
               }

               return result;
            },
            { data: {}, isValid: true },
         );

      const messageStrings = [];
      for (const key in validationResult.data) {
         if (validationResult.data.hasOwnProperty(key)) {
            // eslint-disable-line
            const count = validationResult.data[key];
            if (count > 1) {
               const [cluster, name] = key.split('_');
               messageStrings.push(`Duplicate endpoint set ${name} in cluster ${cluster.toUpperCase()}.`);
            }
         }
      }

      this.setState({ endpointSets: [...endpointSets], validationMessage: messageStrings.join(' ') });
      const isEverySetValid = endpointSets.every(x => x.isValid);

      return isEverySetValid && validationResult.isValid;
   }

   render() {
      const { clusters } = this.props;
      const { validationMessage } = this.state;
      const endpointSetItems = this.state.endpointSets.map(({ cluster, name, isValid, id }) => (
         <BalancersFormEndpointSet
            key={id}
            cluster={cluster}
            clusters={clusters}
            name={name}
            isValid={isValid}
            onClusterChange={([newValue]) => {
               this.onClusterChange(id, newValue);
            }}
            onEndpointSetNameChange={newValue => {
               this.onEndpointSetNameChange(id, newValue);
            }}
            onEndpointSetRemove={() => {
               this.onEndpointSetRemove(id);
            }}
         />
      ));

      return (
         <BalancersFormSection title={'Target endpoint sets'} className={b()}>
            <p>Balancer instances will pass requests to these endpoint sets</p>
            <div className={b('items-container')}>{endpointSetItems}</div>
            {validationMessage && <div className={b('error')}>{validationMessage}</div>}
            <button className={b('add-button')} onClick={this.addEndpointSet.bind(this)} type={'button'}>
               <span className={b('add-button-inner')}>
                  <i className={`far fa-plus-circle ${b('add-icon')}`} />
                  <span className={b('add-text')}>Add endpoint set</span>
               </span>
            </button>
         </BalancersFormSection>
      );
   }
}
