import { ExternalLink } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import React, { Component } from 'react';

import { EXTERNAL_LINKS } from '../../../models';
import { DNSBalancerZones, domains } from '../../store/reducers/balancers';
import BalancersFormSection from '../BalancersFormSection/BalancersFormSection';
import CheckBox from '../CheckBox/CheckBox';
import Select from '../Select/Select';
import TextInput from '../TextInput/TextInput';

import './BalancersFormFQDN.scss';

const b = block('fqdn');

export default class BalancersFormFQDN extends Component {
   state = {
      hostname: '',
      domain: domains[0],
      isCreateDNSRecord: false,
      DNSBalancerZone: DNSBalancerZones[0],
      isValid: true,
   };

   onHostnameChange(hostname) {
      this.setState({ hostname });
   }

   onDomainChange([domain]) {
      this.setState({ domain });
   }

   onDNSCheckboxChange(e) {
      const { checked } = e.target;
      this.setState({ isCreateDNSRecord: checked });
   }

   onDNSZoneChange([DNSBalancerZone]) {
      this.setState({ DNSBalancerZone });
   }

   validate() {
      const { hostname } = this.state;
      const regex = /^[a-z][a-z0-9-.]+$/;
      const isValid = regex.test(hostname);

      this.setState({ isValid });

      return isValid;
   }

   serialize() {
      const { hostname, domain, isCreateDNSRecord, DNSBalancerZone } = this.state;
      const result = {
         id: `${hostname}${domain}`,
         commonName: `${hostname}${domain}`,
      };
      if (isCreateDNSRecord) {
         result.dnsRecordRequest = {
            default: {
               'name_server': {
                  id: DNSBalancerZone,
                  namespaceId: 'infra',
               },
               zone: hostname,
            },
            type: 'DEFAULT',
         };

         if (domain !== `.${DNSBalancerZone}`) {
            result.subjectAlternativeNames = [`${hostname}.${DNSBalancerZone}`];
         }
      }

      return result;
   }

   render() {
      const { hostname, domain, isCreateDNSRecord, DNSBalancerZone, isValid } = this.state;
      return (
         <BalancersFormSection title={'Balancer FQDN'} className={b()}>
            <p>
               Choose FQDN for your balancer wisely, it can’t be changed. But it’s possible to create additional domains
               afterwards.
            </p>
            <p>
               Provided FQDN will be used:
               <br />
               — as an identifier for balancers;
               <br />
               — as a common name for the future TLS certificate;
               <br />— for matching user&#39;s requests.
            </p>
            <div className={b('controls')}>
               <div className={b('control-container')}>
                  <div className={b('control-title')}>Hostname</div>
                  <div className={b('control')}>
                     <TextInput
                        placeholder={'Enter hostname'}
                        text={hostname}
                        onChange={this.onHostnameChange.bind(this)}
                     />
                  </div>
                  <div className={b('error', { hidden: isValid })}>
                     String does not match pattern: ^[a-z][a-z0-9-.]+$
                  </div>
               </div>
               <div className={b('control-container')}>
                  <div className={b('control-title')}>Domain</div>
                  <div className={b('control')}>
                     <Select
                        width={'max'}
                        items={domains.map(x => ({ key: `key-${x}`, val: x, text: x }))}
                        val={domain}
                        onChange={this.onDomainChange.bind(this)}
                     />
                  </div>
               </div>
            </div>
            <div className={b('l7-container')}>
               <CheckBox checked={isCreateDNSRecord} onChange={this.onDNSCheckboxChange.bind(this)}>
                  Create DNS record{' '}
                  <strong>
                     {hostname}.{DNSBalancerZone}
                  </strong>{' '}
                  and point it directly to L7 balancer instances
               </CheckBox>
               <div className={b('text-gray')}>
                  Before <strong>production</strong> use, please:
                  <ul>
                     <li>
                        check DNS load balancing{' '}
                        <ExternalLink href={EXTERNAL_LINKS.wiki.awacsLimitations}>restrictions</ExternalLink>
                     </li>
                     <li>
                        create <ExternalLink href={EXTERNAL_LINKS.wiki.awacsPuncher}>Puncher rules</ExternalLink> for
                        direct access
                     </li>
                  </ul>
               </div>
               <div className={b('control-container', { hidden: !isCreateDNSRecord, dns: true })}>
                  <div className={b('control-title')}>DNS record domain</div>
                  <div className={b('control')}>
                     <Select
                        width={'max'}
                        items={DNSBalancerZones.map(x => ({ key: `key-${x}`, val: x, text: x }))}
                        val={DNSBalancerZone}
                        onChange={this.onDNSZoneChange.bind(this)}
                     />
                  </div>
               </div>
            </div>
         </BalancersFormSection>
      );
   }
}
