import block from 'bem-cn-lite';
import PropTypes from 'prop-types';
import React from 'react';
import CheckBox from '../CheckBox/CheckBox';

import Dialog from '../Dialog/Dialog';

import './ConfirmDialog.scss';

const b = block('confirm-dialog');

export default class ConfirmDialog extends React.Component {
   static propTypes = {
      visible: PropTypes.bool,
      title: PropTypes.string,
      message: PropTypes.node,
      confirmation: PropTypes.node,
      confirmations: PropTypes.arrayOf(PropTypes.node),
      actionId: PropTypes.string,
      actionText: PropTypes.string,
      actionTheme: PropTypes.oneOf(['normal', 'action', 'danger']),
      actionHandler: PropTypes.func,
      cancelText: PropTypes.string,
      onRequestClose: PropTypes.func,
      onSuccess: PropTypes.func,
      onError: PropTypes.func,
   };

   static defaultProps = {
      visible: false,
      actionTheme: 'action',
   };

   state = {
      isLoading: false,
      confirmed: false,
      confirmedArray: {},
   };

   getActions = () => {
      const { confirmation, confirmations, actionText, actionTheme, cancelText, onRequestClose } = this.props;
      const { isLoading, confirmed, confirmedArray } = this.state;

      return [
         {
            text: cancelText,
            handler: onRequestClose,
         },
         {
            text: actionText,
            handler: this.handleActionClick,
            theme: actionTheme,
            progress: isLoading,
            disabled:
               !(Boolean(confirmation) || Boolean(confirmations)) ||
               (Boolean(confirmation) && !confirmed) ||
               (Boolean(confirmations) && confirmations.some((e, i) => !confirmedArray[String(i)])),
         },
      ];
   };

   handleConfirmChange = event => {
      const confirmed = event.target.checked;

      this.setState({ confirmed });
   };

   handleConfirmItemChange = (i, event) => {
      const confirmed = event.target.checked;

      this.setState(state => ({ confirmedArray: { ...state.confirmedArray, [i]: confirmed } }));
   };

   handleActionClick = () => {
      const { actionId, actionHandler, onSuccess, onError, onRequestClose } = this.props;

      if (actionHandler) {
         this.setState({ isLoading: true });

         Promise.resolve(actionHandler())
            .then(() => {
               this.setState({ isLoading: false });

               if (onSuccess) {
                  onSuccess(actionId);
               }

               onRequestClose();
            })
            .catch(error => {
               this.setState({ isLoading: false });

               if (onError) {
                  onError(error);
               }
            });
      }
   };

   renderTitle() {
      const { title } = this.props;

      return <h1 className={b('title')}>{title}</h1>;
   }

   renderMessage() {
      const { message } = this.props;

      if (!message) {
         return null;
      }

      if (typeof message === 'string') {
         return <div className={b('message')} dangerouslySetInnerHTML={{ __html: message }} />;
      }

      return <div className={b('message')}>{message}</div>;
   }

   renderConfirmation() {
      const { confirmation, confirmations } = this.props;
      const { confirmed, confirmedArray } = this.state;

      if (!confirmation && !confirmations) {
         return null;
      }

      if (confirmation) {
         return (
            <div className={b('confirmation')} data-test={'confirmation'}>
               <CheckBox checked={confirmed} onChange={this.handleConfirmChange}>
                  {confirmation}
               </CheckBox>
            </div>
         );
      }

      return (
         <>
            {confirmations.map((confirmationItem, i) => (
               <div key={confirmationItem} className={b('confirmation')} data-test={'confirmation'}>
                  <CheckBox
                     checked={confirmedArray[String(i)]}
                     onChange={this.handleConfirmItemChange.bind(this, String(i))}
                  >
                     {confirmationItem}
                  </CheckBox>
               </div>
            ))}
         </>
      );
   }

   render() {
      const { visible, onRequestClose } = this.props;

      return (
         <Dialog visible={visible} actions={this.getActions()} onOutsideClick={onRequestClose} className={b()}>
            {this.renderTitle()}
            {this.renderMessage()}
            {this.renderConfirmation()}
         </Dialog>
      );
   }
}
