import block from 'bem-cn-lite';

import { Icon, Modal } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component } from 'react';

import Button from '../Button/Button';

import './Dialog.scss';

const b = block('dialog');

export default class Dialog extends Component {
   static propTypes = {
      title: PropTypes.oneOfType([PropTypes.node, PropTypes.func]),
      actions: PropTypes.arrayOf(
         PropTypes.shape({
            theme: PropTypes.oneOf(['normal', 'action', 'danger']),
            text: PropTypes.string,
            handler: PropTypes.func,
            progress: PropTypes.bool,
            disabled: PropTypes.bool,
         }),
      ),
      children: PropTypes.node,
      visible: PropTypes.bool,
      onOutsideClick: PropTypes.func,
      size: PropTypes.oneOf(['s', 'm', 'l', 'full']),
      className: PropTypes.string,
      noAnimation: PropTypes.bool,
   };

   static defaultProps = {
      size: 'm',
      noAnimation: true,
   };

   state = {
      renderCount: 0,
   };

   componentDidMount() {
      requestAnimationFrame(() => this.setState({ renderCount: 1 }));
   }

   renderHeader() {
      const { title } = this.props;

      if (title) {
         return <div className={b('header')}>{typeof title === 'function' ? title() : <h1>{title}</h1>}</div>;
      }

      return null;
   }

   renderBody() {
      const { children } = this.props;

      return <div className={b('body')}>{children}</div>;
   }

   renderFooter() {
      const { actions } = this.props;

      if (actions) {
         return (
            <div className={b('footer')}>
               <div className={b('actions')} data-test={'actions'}>
                  {actions.map(({ theme = 'normal', text, handler, progress, disabled }) => (
                     <div
                        key={text}
                        className={b('action')}
                        data-test={`action-${text.split(' ').join('-').toLowerCase()}`}
                     >
                        <Button
                           view={'default'}
                           tone={'default'}
                           theme={theme}
                           size={'s'}
                           text={text}
                           onClick={disabled ? null : handler}
                           progress={progress}
                           disabled={disabled}
                        />
                     </div>
                  ))}
               </div>
            </div>
         );
      }

      return null;
   }

   renderCloseButton() {
      const { onOutsideClick } = this.props;

      return (
         <div data-test={'button-close'}>
            <Button theme={'clear'} title={'Close'} mix={{ block: b('close') }} onClick={onOutsideClick}>
               <Icon glyph={'type-cross'} />
            </Button>
         </div>
      );
   }

   render() {
      const { visible, onOutsideClick, size, className, noAnimation } = this.props;

      return (
         <Modal
            visible={visible}
            theme={'normal'}
            mix={{ block: `${b({ size })} ${noAnimation ? b('no-animation') : ''}` }}
            onOutsideClick={onOutsideClick}
            autoclosable
            __islandsPopupHack={this.state.renderCount}
         >
            <div className={b('content', className)} data-test={'confirm-dialog'}>
               {this.renderCloseButton()}
               {this.renderHeader()}
               {this.renderBody()}
               {this.renderFooter()}
            </div>
         </Modal>
      );
   }
}
