import block from 'bem-cn-lite';
import { Icon } from 'lego-on-react';
import PropTypes from 'prop-types';
import React from 'react';
import { Link } from 'react-router-dom';

import ActionPopup from '../ActionPopup/ActionPopup';
import Button from '../Button/Button';
import ContainerError from '../ContainerError/ContainerError';
import { Back, More } from '../icons';
import LoadingPage from '../LoadingPage/LoadingPage';
import NoAccessPage from '../NoAccessPage/NoAccessPage';
import Tooltip from '../Tooltip/Tooltip';

import './Page.scss';

const b = block('page');

export default class Page extends React.Component {
   static actionPropType = {
      text: PropTypes.string.isRequired,
      title: PropTypes.string,
      disabled: PropTypes.bool,
      handler: PropTypes.func,
   };

   static primaryActionPropType = {
      ...Page.actionPropType,
      theme: PropTypes.string,
      disabledNote: PropTypes.string,
      progress: PropTypes.bool,
   };

   static propTypes = {
      children: PropTypes.node,
      loaded: PropTypes.bool,
      error: PropTypes.object,
      onErrorAction: PropTypes.func,
      headerExtra: PropTypes.node,
      headerBefore: PropTypes.node,
      title: PropTypes.node,
      primaryActions: PropTypes.oneOfType([
         PropTypes.node,
         PropTypes.arrayOf(PropTypes.shape(Page.primaryActionPropType)),
      ]),
      secondaryActions: PropTypes.arrayOf(PropTypes.shape(Page.actionPropType)),
      className: PropTypes.string,
      dataTest: PropTypes.string,
   };

   static DEFAULT_PRIMARY_ACTION_THEME = 'action';

   state = {
      secondaryActionsPopup: {
         visible: false,
         anchor: null,
      },
   };

   toggleSecondaryActions = event => {
      const target = event.currentTarget;

      this.setState(prevState => ({
         secondaryActionsPopup: {
            anchor: target,
            visible: !prevState.secondaryActionsPopup.visible,
         },
      }));
   };

   closeSecondaryActions = () => {
      this.setState(prevState => ({
         secondaryActionsPopup: {
            ...prevState.secondaryActionsPopup,
            visible: false,
         },
      }));
   };

   handleSecondaryActionClick = actionIndex => {
      const { secondaryActions } = this.props;
      const action = secondaryActions[actionIndex];

      if (action && action.handler) {
         action.handler();
      }

      this.closeSecondaryActions();
   };

   renderBackButton() {
      const { backUrl } = this.props;

      if (!backUrl) {
         return null;
      }

      return (
         <Link to={backUrl} className={b('back-link')}>
            <Back width={14} height={14} fill={'currentColor'} />
         </Link>
      );
   }

   renderTitle() {
      const { title } = this.props;
      let content;
      if (title === undefined) {
         content = null;
      } else if (typeof title === 'string') {
         content = <h1 className={b('title')}>{title}</h1>;
      } else {
         content = <div className={b('title')}>{title}</div>;
      }

      return content;
   }

   renderPrimaryActions() {
      const { primaryActions } = this.props;
      if (!primaryActions) {
         return null;
      }

      return Array.isArray(primaryActions) ? <PrimaryActions actions={primaryActions} /> : primaryActions;
   }

   renderSecondaryActions() {
      const { secondaryActions } = this.props;
      const { secondaryActionsPopup } = this.state;

      if (!secondaryActions) {
         return null;
      }

      const actionsWithId = secondaryActions.map((action, index) => ({ ...action, id: index }));

      return (
         <div className={b('secondary-actions')}>
            <Button
               theme={'clear'}
               mix={{ block: b('secondary-actions-toggle') }}
               onClick={this.toggleSecondaryActions}
            >
               <Icon svg={<More size={16} />} />
            </Button>
            <ActionPopup
               actions={actionsWithId}
               visible={secondaryActionsPopup.visible}
               anchor={secondaryActionsPopup.anchor}
               onOutsideClick={this.closeSecondaryActions}
               onActionClick={this.handleSecondaryActionClick}
            />
         </div>
      );
   }

   renderHeader() {
      const { headerExtra, headerBefore } = this.props;

      return (
         <div className={b('header')}>
            {headerBefore && <div className={b('header-before')}>{headerBefore}</div>}
            <div className={b('header-main')}>
               {this.renderBackButton()}
               {this.renderTitle()}
               {this.renderPrimaryActions()}
               {this.renderSecondaryActions()}
            </div>
            {headerExtra && <div className={b('header-extra')}>{headerExtra}</div>}
         </div>
      );
   }

   renderContent() {
      const { children } = this.props;

      return (
         <div className={b('content')} data-test={'page-content'}>
            {children}
         </div>
      );
   }

   render() {
      const { className, dataTest, loaded, error, onErrorAction } = this.props;
      if (loaded === false) {
         return <LoadingPage />;
      }

      if (error) {
         const { title, description, status } = error;

         switch (status) {
            case 403:
               return <NoAccessPage />;
            default:
               // TODO @seqvirioum: заиспользовать красивую страницу ErrorPage, когда ошибки будут возвращаться в едином формате
               return <ContainerError title={title} description={description} actionCallback={onErrorAction} />;
         }
      }

      return (
         <div className={b(null, className)} data-test={dataTest}>
            {this.renderHeader()}
            {this.renderContent()}
         </div>
      );
   }
}

export function PrimaryActions({ actions }) {
   return (
      <div className={b('primary-actions')} data-test={'primary-actions'}>
         {actions.map(action => (
            <span key={action.text} data-test={`primary-action-${action.text.split(' ').join('-').toLowerCase()}`}>
               <Tooltip active={action.disabled} text={action.disabledNote}>
                  <Button
                     theme={action.theme || Page.DEFAULT_PRIMARY_ACTION_THEME}
                     icon={action.icon}
                     iconLeft={action.iconLeft}
                     iconRight={action.iconRight}
                     text={action.text}
                     attrs={{ title: action.title }}
                     disabled={action.disabled}
                     progress={action.progress}
                     mix={{ block: b('primary-action') }}
                     onClick={action.handler}
                     type={action.type}
                     checked={action.checked}
                     tone={action.tone}
                     view={action.view}
                     pin={action.pin}
                  >
                     <span style={{ position: 'absolute', top: 0, left: -25, width: 28, height: 28 }}>
                        {action.iconLeft ? action.iconLeft : null}
                     </span>
                  </Button>
               </Tooltip>
            </span>
         ))}
      </div>
   );
}

PrimaryActions.propTypes = {
   actions: PropTypes.arrayOf(PropTypes.shape(Page.primaryActionPropType)),
};
