import { ButtonLink } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import { Link, Spin } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { withRouter } from 'react-router';

import { setToListOptions, TagList } from '../../../components/forms/inputs/TagList/TagList';
import { urlBuilder } from '../../../models';
import routes, { createHref } from '../../routes';
import { isMatched } from '../../utils/queryWithTags';

import ArrowToggle from '../ArrowToggle/ArrowToggle';
import MarkedText from '../MarkedText/MarkedText';
import RouteLink from '../RouteLink/RouteLink';
import StageCard from '../StageCard/StageCard';

import './ProjectCard.scss';

const b = block('project-card');

class ProjectCard extends Component {
   static propTypes = {
      stages: PropTypes.object.isRequired,
   };

   state = {
      isLoading: false,
      opened: false,
      search: {
         caption: '',
      },
      filters: {
         stage: '',
      },
      searchCache: {},
   };

   searchCache = {
      stage: {},
   };

   onFiltersReset() {
      this.setState({
         filters: {
            stage: '',
         },
      });
   }

   getStageIdList() {
      const { stageNamesByProject, projectName } = this.props;
      return Object.keys(stageNamesByProject[projectName] || {}) || [];
   }

   renderProjectName(project) {
      return (
         <div className={b('project-name')} data-test={`${project.meta.id}`}>
            {this.renderProjectLink(project)}
         </div>
      );
   }

   renderProjectLink(project) {
      return (
         <RouteLink to={urlBuilder.project(project.meta.id)} theme={'normal'}>
            {project.meta.id}
         </RouteLink>
      );
   }

   renderHeader() {
      const {
         projectName,
         opened,
         globalFilters: { project, objectType, my },
         onProjectHeaderClick,
         objectIdsWithLogins,
         isLoading,
         checkStages,
         startOpened,
         tags,
      } = this.props;
      return (
         <div
            className={b('header', { opened, 'start-opened': startOpened })}
            onClick={() => onProjectHeaderClick(projectName)}
            role={'button'}
            aria-label={'Crate stage'}
            tabIndex={0}
         >
            <div>
               <ArrowToggle direction={opened ? 'top' : 'bottom'} width={11} height={17} />
            </div>
            <div className={b('name')}>
               <RouteLink
                  to={createHref(routes.project, { projectId: projectName || '%empty' })}
                  theme={'normal'}
                  style={{ color: 'black' }}
               >
                  {objectType === 'project' ? (
                     <MarkedText
                        text={projectName || <i>stages without project</i>}
                        searchString={projectName ? project : ''}
                        type={'project'}
                     />
                  ) : (
                     projectName || <i>stages without project</i>
                  )}
               </RouteLink>
               {my !== 'yes' && ((objectIdsWithLogins.project || {})[projectName] || {})[window.USER.login] && (
                  <div className={b('my-label', { accent: true })}>my</div>
               )}
               {tags.length > 0 ? (
                  <TagList editable={false} items={setToListOptions(new Set(tags))} className={b('tags')} />
               ) : null}
               {isLoading && (
                  <div className={b('loading')}>
                     <Spin size={'xxs'} progress />
                  </div>
               )}
               {!isLoading && this.getStageIdList().length === 0 && checkStages && (
                  <div className={b('no-stages-label')}>No stages</div>
               )}
            </div>
            <div className={b('create-stage-button', { visible: opened })}>
               <ButtonLink to={urlBuilder.newStage(projectName)}>Create new stage</ButtonLink>
            </div>
         </div>
      );
   }

   renderStageList() {
      const {
         globalFilters: { stage, my, name, objectType },
         objectIdsWithLogins,
         stages,
         isLoading,
      } = this.props;
      const stageIdList = this.getStageIdList();
      const isMyStage = id => ((objectIdsWithLogins.stage || {})[id] || {})[window.USER.login];
      const isMy = my === 'yes';
      if (stageIdList.length === 0 && !isLoading) {
         return <div style={{ color: 'gray', marginLeft: '40px' }}>No stages</div>;
      }
      return (
         <div className={b('stage-list')}>
            {stageIdList
               .filter(stageId => !isMy || isMyStage(stageId))
               .filter(stageId => {
                  if (objectType === 'stage') {
                     const stageTags = stages[stageId]?.labels?.tags ?? [];

                     return isMatched(name || stage, stageId, stageTags);
                  }

                  return true;
               })
               .sort()
               .map(stageId => (
                  <StageCard
                     key={stageId}
                     stage={stages[stageId]}
                     isMy={((objectIdsWithLogins.stage || {})[stageId] || {})[window.USER.login]}
                     globalFilters={this.props.globalFilters}
                  />
               ))}
         </div>
      );
   }

   renderLoadButton() {
      const { onLoadButtonClick } = this.props;
      return (
         <div style={{ margin: '8px 32px' }}>
            <Link theme={'pseudo'} onClick={() => onLoadButtonClick()}>
               Load More Stages
            </Link>
         </div>
      );
   }

   render() {
      const { projectName, opened, endOpened, startOpened, existMoreStages, checkStages, isLoading } = this.props;
      const noStages = !isLoading && this.getStageIdList().length === 0 && checkStages;
      return (
         <div
            key={projectName}
            className={b({
               'end-opened': endOpened,
               'start-opened': startOpened,
               'single-opened': startOpened && endOpened,
            })}
            data-test={`project--${projectName}`}
         >
            {this.renderHeader()}
            {opened && !noStages && (
               <>
                  {this.renderStageList()}
                  {existMoreStages && this.renderLoadButton()}
               </>
            )}
         </div>
      );
   }
}

export default withRouter(ProjectCard);
