import block from 'bem-cn-lite';
import _isEqual from 'lodash/isEqual';
import _noop from 'lodash/noop';
import PropTypes from 'prop-types';
import React from 'react';

import { KeyCodes } from '../../constants/common';

import './ScrollableList.scss';

const b = block('scrollable-list');

export default class ScrollableList extends React.Component {
   static propTypes = {
      data: PropTypes.array.isRequired,
      itemHeight: PropTypes.number.isRequired,
      renderItem: PropTypes.func.isRequired,
      onAction: PropTypes.func.isRequired,
      visible: PropTypes.bool.isRequired,
      onItemClick: PropTypes.func,
   };

   static defaultProps = {
      onItemClick: _noop,
   };

   state = {
      selectedItemIndex: 0,
   };

   componentDidMount() {
      if (this.props.visible) {
         this.attachKeyDownListeners();
      }
   }

   componentDidUpdate(prevProps) {
      if (!this.props.visible && prevProps.visible) {
         this.detachKeyDownListeners();
      }

      if (this.props.visible && !prevProps.visible) {
         this.attachKeyDownListeners();
         this.moveToCurrentItem(0);
      }

      if (this.props.visible && prevProps.visible) {
         if (!_isEqual(this.props.data, prevProps.data)) {
            this.moveToCurrentItem(0);
         }
      }
   }

   componentWillUnmount() {
      this.detachKeyDownListeners();
   }

   saveContainerRef = ref => {
      this.container = ref;
   };

   handleMouseMove = index => {
      const { selectedItemIndex } = this.state;

      if (selectedItemIndex !== index) {
         this.setState({ selectedItemIndex: index });
      }
   };

   handleKeyDown = event => {
      const { data, onAction } = this.props;
      const { selectedItemIndex } = this.state;

      const itemCount = data.length;

      switch (event.keyCode) {
         case KeyCodes.ARROW_DOWN: {
            event.preventDefault();

            if (selectedItemIndex === itemCount - 1) {
               this.moveToCurrentItem(0);
            } else {
               this.moveToCurrentItem(selectedItemIndex + 1);
            }
            break;
         }
         case KeyCodes.ARROW_UP: {
            event.preventDefault();

            if (selectedItemIndex === 0) {
               this.moveToCurrentItem(itemCount - 1);
            } else {
               this.moveToCurrentItem(selectedItemIndex - 1);
            }
            break;
         }
         case KeyCodes.ENTER: {
            event.preventDefault();

            const selectedItem = data[selectedItemIndex];

            onAction(selectedItem);

            break;
         }
         default:
            break;
      }
   };

   handleClick = index => {
      const { onAction, data } = this.props;
      onAction(data[index]);
   };

   moveToCurrentItem = currentIndex => {
      this.setState({ selectedItemIndex: currentIndex }, this.scrollInnerContainer);
   };

   scrollInnerContainer = () => {
      const { itemHeight } = this.props;
      const { selectedItemIndex } = this.state;

      const offsetCurrentItem = selectedItemIndex * itemHeight;
      const visiblePart = this.container.clientHeight + this.container.scrollTop;
      const visibleItem = offsetCurrentItem + itemHeight;

      if (visibleItem > visiblePart) {
         this.container.scrollTo(0, visibleItem - this.container.clientHeight);
      } else if (offsetCurrentItem < this.container.scrollTop) {
         this.container.scrollTo(0, offsetCurrentItem);
      }
   };

   detachKeyDownListeners() {
      window.removeEventListener('keydown', this.handleKeyDown);
   }

   attachKeyDownListeners() {
      setTimeout(() => {
         window.addEventListener('keydown', this.handleKeyDown);
      }, 0);
   }

   renderListItem = (data, index) => {
      const { renderItem } = this.props;
      const { selectedItemIndex } = this.state;
      const isSelected = selectedItemIndex === index;

      const item = renderItem(data);
      return (
         <div
            key={index}
            className={b('item', { selected: isSelected })}
            onMouseMove={this.handleMouseMove.bind(this, index)}
            onClick={this.handleClick.bind(this, index)}
            role={'button'}
            aria-label={item.toString()}
            tabIndex={0}
            data-test={'list-item'}
         >
            {item}
         </div>
      );
   };

   render() {
      const { data } = this.props;

      return (
         <div className={b()} ref={this.saveContainerRef}>
            {data.map(this.renderListItem)}
         </div>
      );
   }
}
