import block from 'bem-cn-lite';
import noop from 'lodash/noop';
import PropTypes from 'prop-types';
import React from 'react';
import ScrollableList from '../ScrollableList/ScrollableList';

import TextInput from '../TextInput/TextInput';

import './Suggest.scss';

const ITEM_HEIGHT = 50;

const b = block('suggest');

/**
 * Компонент для текстового инпута с садджестом
 *
 * @export
 * @class Suggest
 * @extends {React.Component}
 */
export default class Suggest extends React.Component {
   static propTypes = {
      className: PropTypes.string,
      renderItem: PropTypes.func, // функция-шаблонизатор, которая преобразует объект с данными в html-разметку
      data: PropTypes.arrayOf(
         PropTypes.oneOfType([
            // массив строк или объектов, из которых будут выбираться данные для садджеста
            PropTypes.string,
            PropTypes.object,
         ]),
      ).isRequired,
      disabled: PropTypes.bool,
      isCommaSeparated: PropTypes.bool,
      separator: PropTypes.string,
      filter: PropTypes.bool, // флаг, указывающий, нужно ли фильтровать садджест-айтемы в соответствии с тем, что введено в инпуте
      filterBy: PropTypes.string, // to make filtering work, you have to specify this if data is array of objects
      placeholder: PropTypes.string,
      onChange: PropTypes.func, // внешний обработчик, который будет дергаться, когда в текстбоксе будет изменяться значение
      onAction: PropTypes.func, // внешний обработчик, который будет дергаться, когда пользователь делает клик на одном из элементов листа
      value: PropTypes.oneOfType([
         // строка или объект, который хранит выбранное значение
         PropTypes.string,
         PropTypes.object,
      ]),
      filterValueBy: PropTypes.string, // если в списке элементов и в valuе cодержится объект, то в text будет выводиться значение поля, указанное в filterValueBy
      group: PropTypes.string, // прилетает из TVM'а и пробрасывается в onChange. TODO: отрефакторить и выпилить
      actionValueFormatter: PropTypes.func, // вызывается, когда перед отображением в текстовом виде нужно как-то преобразовать выбранный элемент. Например 'foo' -> 'foo='
      onChangeOmnibarVisibilityCheckCallback: PropTypes.func, // вызывается в onChange, принимает новое значение text. Возвращает true, если нужно показать элементы садджеста
      attrs: PropTypes.object, // доп. html-атрибуты, которые будут навешаны на контейнер садджеста
      showFirst: PropTypes.number,
   };

   static defaultProps = {
      onChange: noop,
      onAction: noop,
      isCommaSeparated: false,
      separator: ',',
      placeholder: '',
      renderItem: x => x, // дефолтная шаблонизация
      attrs: {},
   };

   constructor(props) {
      const { filterValueBy, value } = props;
      super(props);
      this.state = {
         text: value[filterValueBy] ? value[filterValueBy] : value,
         omnibarVisible: false,
      };
      this.inputRef = React.createRef();
      this.containertRef = React.createRef();
   }

   componentDidUpdate(prevProps, prevState) {
      if (this.state.omnibarVisible && !prevState.omnibarVisible) {
         this.attachClickListeners();
      } else if (!this.state.omnibarVisible && prevState.omnibarVisible) {
         this.detachClickListeners();
      }
   }

   componentWillUnmount() {
      this.detachClickListeners();
   }

   handleOutsideClick = event => {
      const container = this.containertRef.current;
      if (container && !container.contains(event.target)) {
         this.hideOmnibar();
         if (this.props.noCustomValue) {
            this.setState({
               text: '',
            });
         }
      }
   };

   /**
    * Срабатывает, когда пользователь делает клик на элементе списка садджеста
    *
    * @memberof Suggest
    */
   onItemClick = item => {
      if (!item) {
         return;
      }
      const { filterValueBy, isCommaSeparated, separator, actionValueFormatter } = this.props;
      const valueItems = isCommaSeparated
         ? // eslint-disable-next-line react/no-access-state-in-setstate
           this.state.text
              .split(separator)
              .map(chunk => chunk.trim())
              .filter(chunk => chunk)
         : [];

      let text;

      if (typeof item === 'string') {
         text = actionValueFormatter ? actionValueFormatter(item) : item;
      } else if (filterValueBy && item[filterValueBy]) {
         text = actionValueFormatter ? actionValueFormatter(item[filterValueBy]) : item[filterValueBy];
      }

      if (text) {
         valueItems.push(text);
      }

      this.setState({ text: valueItems.join(`${separator} `), omnibarVisible: false }, () => {
         if (isCommaSeparated) {
            this.props.onAction(valueItems.join(`${separator} `));
         } else {
            this.props.onAction(item);
         }
      });

      this.inputRef.focus();
   };

   handleFocus = () => {
      if (!this.state.omnibarVisible) {
         this.showOmnibar();
      }
   };

   /**
    * Обработчик изменения текста в текстовом поле
    *
    * @memberof Suggest
    */
   handleChange = text => {
      const { onChange, group, onChangeOmnibarVisibilityCheckCallback } = this.props;

      // Требуется разорвать event-loop, чтобы не фризить рендер
      const onChangeDelayed = (...args) => window.setTimeout(() => onChange(...args));

      this.setState({ text }, () => (group ? onChangeDelayed(text, group) : onChangeDelayed(text))); // TODO: выпилить group и тернарник

      if (onChangeOmnibarVisibilityCheckCallback) {
         this.setState({ omnibarVisible: onChangeOmnibarVisibilityCheckCallback(text) });
      } else {
         this.setState({ omnibarVisible: true });
      }
   };

   /**
    * Возвращает отфильтрованный список элементов для садджеста
    *
    * @returns
    * @memberof Suggest
    */
   getListItems() {
      const { data, filter, filterBy } = this.props;
      const { text } = this.state;

      if (!data.length) {
         return [];
      }

      if (!filter) {
         return data;
      }

      if (typeof data[0] === 'string') {
         return data.filter(item => item.toLowerCase().includes(text.toLowerCase()));
      }

      if (!filterBy) {
         return data;
      }

      return data.filter(item => {
         const value = item[filterBy];
         if (!value) {
            return false;
         }

         return value.toLowerCase().includes(text.toLowerCase());
      });
   }

   showOmnibar = () => {
      this.setState({ omnibarVisible: true });
   };

   hideOmnibar = () => {
      this.setState({ omnibarVisible: false });
   };

   detachClickListeners() {
      window.removeEventListener('click', this.handleOutsideClick);
   }

   attachClickListeners() {
      window.addEventListener('click', this.handleOutsideClick);
   }

   render() {
      const { omnibarVisible, text } = this.state;
      const { renderItem, disabled, placeholder, showFirst } = this.props;

      const allData = this.getListItems();
      const data = showFirst ? allData.slice(0, showFirst) : allData;

      return (
         <div className={b()} ref={this.containertRef}>
            <TextInput
               hasClear
               text={text}
               type={'text'}
               innerRef={el => {
                  this.inputRef = el;
               }}
               disabled={disabled}
               onChange={this.handleChange}
               onFocus={this.handleFocus}
               placeholder={placeholder}
            />
            <div className={b('omnibar', { visible: omnibarVisible })}>
               <ScrollableList
                  data={data}
                  renderItem={renderItem}
                  onAction={this.onItemClick}
                  itemHeight={ITEM_HEIGHT}
                  visible={omnibarVisible}
               />
            </div>
         </div>
      );
   }
}
