import block from 'bem-cn-lite';

import { Popup } from 'lego-on-react';
import PropTypes from 'prop-types';
import React from 'react';

import './Tooltip.scss';

const b = block('tooltip');

export default class Tooltip extends React.Component {
   static propTypes = {
      active: PropTypes.bool,
      text: PropTypes.oneOfType([PropTypes.string, PropTypes.node]),
      children: PropTypes.node,
      visible: PropTypes.bool,
      alwaysVisible: PropTypes.bool,
   };

   static defaultProps = {
      active: true,
      alwaysVisible: false,
   };

   static directions = ['bottom-center', 'bottom-left', 'bottom-right', 'top-center', 'top-left', 'top-right'];

   static SHOW_TIMEOUT = 150;

   static HIDE_TIMEOUT = 150;

   state = {
      textContainerWidth: null,
      popupVisible: false,
   };

   componentDidMount() {
      const { alwaysVisible } = this.props;
      if (this.popupTextRef) {
         // offsetWidth is returning a rounded down value, it can cause an ugly effect,
         // add 1px to compensate possible roundness
         const width = this.popupTextRef.offsetWidth + 1;

         this.setState({
            textContainerWidth: width,
         });
      }
      if (alwaysVisible) {
         window.setTimeout(this.handleMouseEnter.bind(this), 1000);
      }
   }

   componentWillUnmount() {
      this.clearHideTimeout();
      this.clearShowTimeout();
   }

   savePopupTextRef = ref => {
      this.popupTextRef = ref;
   };

   saveContentRef = ref => {
      this.contentRef = ref;
   };

   clearShowTimeout = () => {
      if (this.showTimeoutId) {
         clearTimeout(this.showTimeoutId);
         this.showTimeoutId = null;
      }
   };

   clearHideTimeout = () => {
      if (this.hideTimeoutId) {
         clearTimeout(this.hideTimeoutId);
         this.hideTimeoutId = null;
      }
   };

   handleMouseClick = () => {
      if (this.state.popupVisible) {
         this.handleMouseLeave();
      } else {
         this.handleMouseEnter();
      }
   };

   handleMouseEnter = () => {
      this.clearHideTimeout();

      this.showTimeoutId = setTimeout(() => {
         this.setState({ popupVisible: true });
      }, Tooltip.SHOW_TIMEOUT);
   };

   handleMouseLeave = () => {
      this.clearShowTimeout();

      this.hideTimeoutId = setTimeout(() => {
         this.setState({ popupVisible: false });
      }, Tooltip.HIDE_TIMEOUT);
   };

   onOutsideClick = () => {
      this.setState(prevState => ({
         popupVisible: !prevState.popupVisible,
      }));
   };

   renderPopup() {
      const { text, hasTail, directions, autoclosable, clickable, mix, buttonClose, modal, visible } = this.props;
      const { popupVisible, textContainerWidth } = this.state;
      const textContainerStyle = {
         width: textContainerWidth,
      };

      return (
         <Popup
            theme={'normal'}
            visible={visible === undefined ? popupVisible : visible}
            anchor={this.contentRef}
            directions={directions || Tooltip.directions}
            hasTail={hasTail}
            autoclosable={autoclosable || !clickable}
            onOutsideClick={modal ? () => {} : this.onOutsideClick}
            mix={{ block: b('popup-body', { [mix]: mix !== undefined }) }}
         >
            <div style={textContainerStyle} className={b('popup-content', { [mix]: mix !== undefined })}>
               <span className={b('popup-text')} ref={this.savePopupTextRef}>
                  {text}
               </span>
               {buttonClose ? (
                  <button className={b('popup-close')} onClick={this.handleMouseLeave} type={'button'}>
                     <i className={'far fa-times'} />
                  </button>
               ) : null}
            </div>
         </Popup>
      );
   }

   renderContent() {
      const { children, clickable } = this.props;

      return (
         <div
            className={b('content')}
            ref={this.saveContentRef}
            {...(clickable
               ? {
                    onClick: event => {
                       event.preventDefault();
                       this.handleMouseClick();
                    },
                 }
               : {
                    onMouseEnter: this.handleMouseEnter,
                    onMouseLeave: this.handleMouseLeave,
                 })}
         >
            {children}
         </div>
      );
   }

   // TODO @seqvirioum: переключиться на Fragment в React 16
   render() {
      const { text, children, active } = this.props;

      return active && text ? (
         <div className={b()}>
            {this.renderContent()}
            {this.renderPopup()}
         </div>
      ) : (
         children
      );
   }
}
