import { ExternalLink } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';
import { createStructuredSelector } from 'reselect';
import { EXTERNAL_LINKS, urlBuilder } from '../../../models';

import { StageNotFound } from '../../../pages/stages/_stageId/components';
import { getStages, selectStage } from '../../../redux';

import { Breadcrumbs } from '../../components/Breadcrumbs/Breadcrumbs';
import ConfirmDialog from '../../components/ConfirmDialog/ConfirmDialog';
import withNotifications from '../../components/hoc/withNotifications';
import NewBalancerForm from '../../components/NewBalancerForm/NewBalancerForm';
import Page from '../../components/Page/Page';
import PageWrapper from '../../components/PageWrapper/PageWrapper';
import routes, { createHref } from '../../routes';
import { createBalancer } from '../../store/reducers/balancers';
import { getClusters } from '../../store/reducers/clusters';

import './BalancerCreatePage.scss';

const b = block('balancer-create-page');

class BalancerCreatePage extends Component {
   static propTypes = {
      history: PropTypes.object,
      match: PropTypes.object,
      abcServiceId: PropTypes.string,
      stage: PropTypes.object,
      notifications: PropTypes.object.isRequired,
      clusters: PropTypes.array,
   };

   state = { isFormProcessing: false };

   componentDidMount() {
      const {
         getStage,
         match: {
            params: { stageId },
         },
      } = this.props;

      getStage(stageId);
   }

   componentWillUnmount() {
      const { notifications } = this.props;
      Object.keys(notifications.notificationsData).forEach(notificationId => {
         notifications.remove(notificationId);
      });
   }

   onFormSubmit = async (
      login,
      serviceId,
      alertRecipientsGroupId,
      fqdnData,
      endpointSetsData,
      balancerInstancesData,
   ) => {
      const {
         history,
         match: {
            params: { stageId },
         },
         notifications,
      } = this.props;

      const confirmAction = async () => {
         this.setState({ isFormProcessing: true });
         const { isSuccess, message, title } = await createBalancer(
            stageId,
            login,
            serviceId,
            alertRecipientsGroupId,
            fqdnData,
            endpointSetsData,
            balancerInstancesData,
         );
         if (isSuccess) {
            history.push(createHref(routes.namespaceInfo, { namespaceId: fqdnData.id }));
         } else {
            notifications.add({
               level: 'error',
               title,
               message,
               autoDismiss: 0,
            });
         }
         this.setState({ isFormProcessing: false });
      };

      this.openConfirmDialog({
         title: 'Confirm new balancer',
         message: (
            <>
               You are going to create a new balancer. Deleting it is a non-trivial process and can cause difficulties.
               See <ExternalLink href={EXTERNAL_LINKS.wiki.removeBalancer}>instructions</ExternalLink>.
            </>
         ),
         actionText: 'Create',
         actionTheme: 'action',
         actionHandler: confirmAction,
         confirmation: "Yes, I know what I'm doing",
         cancelText: 'Cancel',
      });
   };

   openConfirmDialog(confirmDialog) {
      this.setState({ confirmDialog });
   }

   renderBreadcrumbs() {
      const {
         match: {
            params: { stageId },
         },
      } = this.props;

      return (
         <Breadcrumbs
            links={[
               { name: 'All', url: urlBuilder.projects() },
               { name: stageId, url: urlBuilder.stage(stageId) },
               { name: 'Balancers', url: urlBuilder.stageBalancers(stageId) },
               { name: 'New' },
            ]}
         />
      );
   }

   renderConfirmDialog() {
      const { confirmDialog } = this.state;

      if (!confirmDialog) {
         return null;
      }

      return (
         <ConfirmDialog
            visible={true}
            title={confirmDialog.title}
            message={confirmDialog.message}
            actionText={confirmDialog.actionText}
            actionTheme={confirmDialog.actionTheme}
            actionHandler={confirmDialog.actionHandler}
            cancelText={confirmDialog.cancelText}
            confirmation={confirmDialog.confirmation}
            onRequestClose={() => {
               this.setState({ confirmDialog: null });
            }}
         />
      );
   }

   render() {
      const {
         match: {
            params: { stageId },
         },
         stage,
         clusters,
      } = this.props;

      if (!stage) {
         // для урлов с несуществующим стейджом
         // /balancers/create/stage-not-found
         return <StageNotFound stageId={stageId} />;
      }

      const { isFormProcessing } = this.state;
      const { spec: { account_id = '' } = {} } = stage;
      const abcServiceIdStr = account_id.split(':').pop();
      const abcServiceId = parseInt(abcServiceIdStr, 10);
      const title = <h1>New Balancer</h1>;

      return (
         <PageWrapper title={`New balancer for ${stageId}`}>
            <Page title={title} headerBefore={this.renderBreadcrumbs()} backUrl={urlBuilder.stageBalancers(stageId)}>
               <div className={b()}>
                  <NewBalancerForm
                     stageId={stageId}
                     abcServiceId={Number.isNaN(abcServiceId) ? null : abcServiceId}
                     onFormSubmit={this.onFormSubmit}
                     isFormProcessing={isFormProcessing}
                     project={[stage]}
                     clusters={clusters}
                  />
               </div>
               {this.renderConfirmDialog()}
            </Page>
         </PageWrapper>
      );
   }
}

const mapDispatchToProps = dispatch => ({
   getStage: stageId => dispatch(getStages({ objectIds: [stageId] })),
});

const mapStateToProps = createStructuredSelector({
   stage: (state, ownProps) => selectStage(state, ownProps.match.params.stageId),
   clusters: state => getClusters(state).map(x => x.value.replace('-', '_')),
});

export default compose(withRouter, withNotifications, connect(mapStateToProps, mapDispatchToProps))(BalancerCreatePage);
