import block from 'bem-cn-lite';
import { Spin } from 'lego-on-react';
import { debounce, get, set } from 'lodash';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';

import spacings from '../../components/DesignSystem/InfracloudSpacings/InfracloudSpacings';
import InfrauiContainer from '../../components/DesignSystem/InfrauiContainer/InfrauiContainer';
// import Button from '../../components/Button/Button';
import withNotifications from '../../components/hoc/withNotifications';
import RouteLink from '../../components/RouteLink/RouteLink';
import Suggest from '../../components/Suggest/Suggest';
import { WindowTitle } from '../../components/WindowTitle/WindowTitle';
import routes, { createHref } from '../../routes';
import { fetchAllNamespaces } from '../../store/reducers/balancers';
import { fetchPeople } from '../../store/reducers/staff';

import './Balancers.scss';

const b = block('balancers');
const bc = block('balancers-container');

class Balancers extends Component {
   static propTypes = {
      dispatch: PropTypes.func,
      namespaces: PropTypes.array,
   };

   state = {
      namespaceTree: {},
      visibility: null,
      filter: '',
   };

   setVisibilityTreeDebounced = debounce(this.setVisibilityTree, 300).bind(this);

   componentDidMount() {
      const { dispatch } = this.props;
      const currentUser = window.USER.login;

      dispatch(fetchPeople({ params: { logins: [currentUser] } }));
      dispatch(fetchAllNamespaces());
   }

   componentDidUpdate(prevProps) {
      const { namespaces } = this.props;
      const { namespaces: prevNamespaces } = prevProps;

      if (namespaces !== prevNamespaces) {
         this.setNamespaceTree();
      }
   }

   setVisibilityTree(filter) {
      const { namespaces } = this.props;
      if (!filter) {
         this.setState({ visibility: null, filter: '' });
         return;
      }

      const visibleNamespaces = namespaces.filter(namespace => namespace.meta.id.includes(filter));
      const visibility = visibleNamespaces.reduce((result, namespace) => {
         const {
            meta: { category },
         } = namespace;
         const chunks = category.split('/');
         chunks.forEach(chunk => {
            result[chunk] = true;
         });

         return result;
      }, {});

      this.setState({ visibility, filter });
   }

   setNamespaceTree() {
      const { namespaces = [] } = this.props;
      const namespaceTree = {};
      const { filter } = this.state;

      const namespacesFiltered = filter
         ? namespaces.filter(currentNamespace => {
              const {
                 meta: { category, id },
              } = currentNamespace;

              return category.includes(filter) || id.includes(filter);
           })
         : namespaces;

      namespacesFiltered.forEach(currentNamespace => {
         const {
            meta: { category },
         } = currentNamespace;

         const path = category.replace(/\//g, '.');
         const item = get(namespaceTree, path);

         if (!item) {
            set(namespaceTree, path, { '--namespaces': [currentNamespace] });
         } else if (item['--namespaces']) {
            item['--namespaces'].push(currentNamespace);
         } else {
            item['--namespaces'] = [currentNamespace];
         }
      }, {});

      Object.keys(namespaceTree).forEach(key => {
         const node = namespaceTree[key];
         this.setCategoryForNode(node, key);
      });

      this.setState({ namespaceTree });
   }

   setCategoryForNode(node, nodeKey) {
      const categoryKeys = Object.keys(node).filter(key => key !== '--namespaces');

      categoryKeys.forEach(key => {
         const nestedNode = node[key];

         this.setCategoryForNode(nestedNode, key);
      });
      node['--category'] = nodeKey;
   }

   renderControls() {
      const { namespaces = [] } = this.props;
      const dataItems = namespaces.map(namespace => namespace.meta.id);

      return (
         <div className={b('controls')}>
            <RouteLink
               to={createHref(routes.balancers)}
               theme={'normal'}
               cls={b('breadcrumbs', null, spacings({ 'indent-b': 'm' }))}
            >
               All namespaces
            </RouteLink>
            <div className={b('suggest-container')}>
               <Suggest
                  data={dataItems}
                  filter={true}
                  onChange={this.setVisibilityTreeDebounced}
                  onAction={this.setVisibilityTreeDebounced}
                  value={''}
                  placeholder={'Filter namespaces by name or by category'}
                  showFirst={25}
               />
               {/* <Button
                        onClick={() => { this.setVisibilityTreeDebounced(''); }}
                        cls={b('reset-btn', null, spacings({'indent-l': 'l'}))}
                    >
                        Reset
                    </Button> */}
            </div>
         </div>
      );
   }

   renderContent() {
      const { namespaces } = this.props;
      const { namespaceTree } = this.state;

      if (!namespaces) {
         return (
            <div className={b('spinner')}>
               <Spin size={'l'} progress />
            </div>
         );
      }

      const topLevelCategories = Object.values(namespaceTree);
      return (
         <div className={b('content', null, spacings({ 'space-t': 'xl' }))}>
            {topLevelCategories.map(this.renderNode)}
         </div>
      );
   }

   renderNode = node => {
      const { visibility, filter } = this.state;
      const category = node['--category'] || 'No category';
      const namespaces = node['--namespaces'] || [];
      const namesapceFiltered = namespaces.filter(
         ({ meta: { id } }) => !visibility || category.includes(filter) || id.includes(filter),
      );
      const nestedCategories = Object.keys(node)
         .filter(key => key !== '--category' && key !== '--namespaces')
         .map(key => node[key]);
      const isHidden = visibility && !visibility[category];
      const isExpanded = Boolean(visibility) && namesapceFiltered.length < 5;

      return (
         <InfrauiContainer
            header={this.renderHeader(category)}
            headerSpacings={{ 'indent-b': 'xs' }}
            spacings={{ 'indent-b': 'm', 'space-h': 'xl' }}
            isInteractive={true}
            isExpanded={isExpanded}
            className={bc({ hidden: isHidden })}
            data-test={'category-container'}
         >
            {namesapceFiltered.length ? (
               <div className={bc('namespaces')}>{this.renderNamespaces(namesapceFiltered)}</div>
            ) : null}
            <div className={bc('nestedNodes')}>{this.renderNestedCategories(nestedCategories)}</div>
         </InfrauiContainer>
      );
   };

   renderNamespaces(namespaces) {
      return namespaces.map(namespace => {
         const {
            meta: { id },
         } = namespace;

         return (
            <RouteLink
               key={id}
               to={createHref(routes.namespaceInfo, { namespaceId: id })}
               theme={'normal'}
               cls={bc('namespace', null, spacings({ 'indent-b': 'xs' }))}
            >
               {id}
            </RouteLink>
         );
      });
   }

   renderNestedCategories(nestedCategories) {
      return nestedCategories.map(this.renderNode);
   }

   renderHeader(category, namespacesCount) {
      return (
         <div className={bc('header')}>
            {category} {namespacesCount ? <span className={bc('header-count')}>({namespacesCount})</span> : null}
         </div>
      );
   }

   render() {
      return (
         <WindowTitle title={'Balancers'}>
            <div className={b()} data-test={'balancers-list'}>
               <InfrauiContainer spacings={{ 'space-v': 'xl', 'space-h': '2xl' }}>
                  {this.renderControls()}
                  {this.renderContent()}
               </InfrauiContainer>
            </div>
         </WindowTitle>
      );
   }
}

const mapStateToProps = state => {
   const {
      balancers: { namespaces },
   } = state;

   return {
      namespaces,
   };
};

export default compose(withRouter, withNotifications, connect(mapStateToProps))(Balancers);
