import { ButtonLink, EmptyContainer, EmptyContainerType } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import { Spin, UserAccount } from 'lego-on-react';
import { get, uniq } from 'lodash';
import moment from 'moment';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';

import { urlBuilder } from '../../../models/urlBuilder';

import { Breadcrumbs } from '../../components/Breadcrumbs/Breadcrumbs';
import InfracloudIconButton from '../../components/DesignSystem/InfracloudIconButton/InfracloudIconButton';
import IconPlus from '../../components/DesignSystem/InfracloudIconPlusText/InfracloudIconPlusText';
import InfracloudOwners from '../../components/DesignSystem/InfracloudOwners/InfracloudOwners';
import spacings from '../../components/DesignSystem/InfracloudSpacings/InfracloudSpacings';
import InfracloudStatusLights from '../../components/DesignSystem/InfracloudStatusLights/InfracloudStatusLights';
import InfracloudTable from '../../components/DesignSystem/InfracloudTable/InfracloudTable';
import InfrauiContainer from '../../components/DesignSystem/InfrauiContainer/InfrauiContainer';
import withNotifications from '../../components/hoc/withNotifications';
import Link from '../../components/Link/Link';
import Page from '../../components/Page/Page';
import PageWrapper from '../../components/PageWrapper/PageWrapper';
import Status from '../../components/Status/Status';
import routes, { createHref } from '../../routes';
import { fetchNamespace, getUnifiedStatus, stopBalancersFetch } from '../../store/reducers/balancers';

import './NamespaceInfoPage.scss';
import { EXTERNAL_LINKS } from '../../../models';

const b = block('namespace-info');

const ExternalLink = props => (
   <Link url={props.url}>
      {props.text}
      {props.hideIcon ? null : <i className={`far fa-external-link-square-alt ${spacings({ 'indent-l': '2xs' })}`} />}
   </Link>
);

const NANNY_UI_DOMAIN = window.CONFIG.nannyUiDomain; // TODO use getConfig()

class NamespaceInfoPage extends Component {
   static propTypes = {
      namespaceId: PropTypes.string,
      history: PropTypes.object,
      match: PropTypes.object,
      namespace: PropTypes.object,
      abcServices: PropTypes.object,
      staffGroups: PropTypes.array,
      dispatch: PropTypes.func,
      updateTime: PropTypes.object,
   };

   static defaultProps = {
      abcServices: {},
   };

   componentDidMount() {
      this.updateNamespace();

      const header = document.querySelector('.page__header');
      if (header) {
         header.style.setProperty('border', 'none');
      }
   }

   componentWillUnmount() {
      const { dispatch } = this.props;
      dispatch(stopBalancersFetch());

      const header = document.querySelector('.page__header');
      if (header) {
         header.style.removeProperty('border');
      }
   }

   updateNamespace = () => {
      const {
         match: {
            params: { namespaceId },
         },
         dispatch,
      } = this.props;

      dispatch(fetchNamespace(namespaceId));
   };

   getRevisionStatus = revisionStatus => {
      const { active = {}, inProgress = {}, validated = {}, id } = revisionStatus;

      const balancerIds = Object.keys(validated);

      if (!balancerIds.length) {
         return null;
      }

      const status = balancerIds
         .map(balancerId => {
            const statusData = {
               validated: validated[balancerId] || { status: 'False' },
               inProgress: inProgress[balancerId] || { status: 'False' },
               active: active[balancerId] || { status: 'False' },
            };

            const unifiedStatus = getUnifiedStatus(statusData);

            return {
               balancerId,
               status: unifiedStatus,
            };
         })
         .reduce((result, statusData) => {
            const key = statusData.status.text;
            if (result[key]) {
               result[key].push(statusData);
            } else {
               result[key] = [statusData];
            }
            return result;
         }, {});

      return {
         revisionId: id,
         status,
      };
   };

   getBackendBalancerStatusMarkup = (balancerStatuses, keyPrefix) => {
      if (!balancerStatuses || !Object.keys(balancerStatuses.status)) {
         return String.fromCharCode(0x2014);
      }

      const items = Object.keys(balancerStatuses.status).map(text => {
         const balancersByStatus = balancerStatuses.status[text];
         let state;
         if (keyPrefix.includes('dns') && text === 'Validated') {
            // TODO: Подумать, как выпилить это говно
            state = 'ok';
         } else {
            state = balancersByStatus[0].status.status;
         }
         return (
            <Status key={`${keyPrefix}-${text}`} type={'secondary'} state={state}>
               {balancersByStatus.length} {text}
            </Status>
         );
      });

      return items;
   };

   getAllRevisionsStatuses = (dnsRecordStatuses = [], l3Statuses = [], statuses = [], id, currentRevisionId) => {
      /* eslint-disable no-shadow */ // TODO
      const revisionsObject = {};

      dnsRecordStatuses.forEach(({ id, ctime }) => {
         revisionsObject[id] = ctime;
      });
      l3Statuses.forEach(({ id, ctime }) => {
         revisionsObject[id] = ctime;
      });
      statuses.forEach(({ id, ctime }) => {
         revisionsObject[id] = ctime;
      });

      const revisionIds = currentRevisionId
         ? [currentRevisionId].concat(
              Object.keys(revisionsObject).filter(revisionId => revisionId !== currentRevisionId),
           )
         : Object.keys(revisionsObject);

      return revisionIds.map(revisionId => {
         const dnsData = dnsRecordStatuses.find(({ id }) => id === revisionId);
         const dnsStatus = dnsData && this.getRevisionStatus(dnsData);
         const l3Data = l3Statuses.find(({ id }) => id === revisionId);
         const l3Status = l3Data && this.getRevisionStatus(l3Data);
         const l7Data = statuses.find(({ id }) => id === revisionId);
         const l7Status = l7Data && this.getRevisionStatus(l7Data);

         return {
            id,
            revisionId,
            dnsStatus,
            l3Status,
            l7Status,
            endpointSets: [],
            revisionCtime: revisionsObject[revisionId],
         };
      });
      /* eslint-enable no-shadow */
   };

   createRevisionMarkup = (revisionData, version) => {
      const {
         id,
         revisionId,
         dnsStatus,
         l3Status,
         l7Status,
         mtime,
         revisionCtime,
         author,
         deleted,
         url,
      } = revisionData;

      const time = mtime || revisionCtime;
      const idMarkup = url ? (
         <ExternalLink
            url={url}
            hideIcon={true}
            text={
               <div className={b('backend-id', { version })}>
                  {id} {version && <span className={b('backend-id-hint')}>({`${version} version`})</span>}
               </div>
            }
         />
      ) : (
         <div className={b('backend-id', { version })}>
            {id} {version && <span className={b('backend-id-hint')}>({`${version} version`})</span>}
         </div>
      );

      return {
         id: idMarkup,
         l7Status: this.getBackendBalancerStatusMarkup(l7Status, `${id}-l7`),
         l3Status: this.getBackendBalancerStatusMarkup(l3Status, `${id}-l3`),
         dnsStatus: this.getBackendBalancerStatusMarkup(dnsStatus, `${id}-dns`),
         revision: revisionId.split('-').shift(),
         modified: moment(time).fromNow(),
         author: author ? (
            <UserAccount
               uid={author}
               name={author}
               avatarId={author}
               hasAccentLetter={true}
               url={`https://staff.yandex-team.ru/${author}`}
               avatarHost={'https://center.yandex-team.ru'}
               provider={'yandex-team'}
               cls={b('user-account')}
            />
         ) : (
            ''
         ),
         related: !version || version === 'current' ? 'N/A' : null,
         deleted: deleted ? <i className={`far fa-check ${b('check-icon')}`} /> : null,
      };
   };

   renderResolveStatus = resolveStatus => {
      if (resolveStatus) {
         const stateMap = {
            'True': 'ok',
            'False': 'error',
         };

         const textMap = {
            'True': 'Success',
            'False': 'Invalid',
         };

         return (
            <Status type={'secondary'} state={stateMap[resolveStatus] || 'unknown'}>
               {textMap[resolveStatus] || 'Unknown'}
            </Status>
         );
      }
      return String.fromCharCode(0x2014);
   };

   onIconClick = (entity, action, entityId) => {
      const {
         match: {
            params: { namespaceId },
         },
      } = this.props;
      const namespaceURL = `${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}`;
      const entityURLMapping = {
         dns: `/dns-records/list/${entityId}`,
         cert: `/certs/list/${entityId}`,
         l3: `/l3-balancers/list/${entityId}`,
         l7: `/balancers/list/${entityId}`,
         upstream: `/upstreams/list/${entityId}`,
         backend: `/backends/list/${entityId}`,
         endpointSet: `/endpoint-sets/list/${entityId}`,
         namespace: '',
      };
      const actionURLMapping = {
         view: '/show/',
         edit: '/edit/',
         delete: '/delete/',
         history: '/history/',
         monitoring: '/monitoring/',
         its: '/its/',
         map: '/map/',
         alert: '/alerting/',
      };
      const url = `${namespaceURL}${entityURLMapping[entity]}${actionURLMapping[action]}`;

      window.open(url, '_blank');
   };

   onAddClick = entity => {
      const {
         match: {
            params: { namespaceId },
         },
      } = this.props;
      const namespaceURL = `${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}`;

      const entityURLMapping = {
         dns: '/dns-records/create/',
         cert: '/certs/order/',
         'existing-L3': '/l3-balancers/add/',
         'new-L3': '/l3-balancers/create/',
         'new-L7': '/balancers/create/',
         'existing-L7': '/balancers/add-from-service/',
         'new-upstream': '/upstreams/create/',
         'existing-upstream': '/upstreams/list/',
         'new-backend': '/backends/create/',
         'existing-backend': '/backends/list/',
      };

      const url = `${namespaceURL}${entityURLMapping[entity]}`;

      window.open(url, '_blank');
   };

   createAddButton(action, text) {
      return (
         <button className={b('add-btn', null, spacings({ 'indent-r': 'xl' }))} onClick={action} type={'button'}>
            <IconPlus text={text} icon={'fa fa-plus'} />
         </button>
      );
   }

   renderSectionTitle(title, total, active, inProgress, error) {
      return (
         <div className={b('section-title')}>
            <span className={b('section-title-text', null, spacings({ 'indent-r': 'xl' }))}>{title}</span>
            <InfracloudStatusLights
               isAnimated={false}
               total={total}
               active={active}
               progress={inProgress}
               error={error}
            />
         </div>
      );
   }

   renderControls(entityKey, controlKeyAttr, ...callbackAttrs) {
      const controlsMapping = {
         'dns': ['view', 'delete'],
         'cert': ['view', 'delete'],
         'l3': ['view', 'history', 'edit', 'delete'],
         'l7': ['view', 'monitoring', 'its', 'history', 'edit', 'delete'],
         'upstream': ['view', 'history', 'edit', 'delete'],
         'backend': ['view', 'history', 'edit', 'delete'],
         'endpointSet': ['view', 'history'],
      };

      const iconClassMapping = {
         'view': 'fa-search',
         'delete': 'fa-trash-alt',
         'history': 'fa-history',
         'edit': 'fa-pencil',
         'freeze': 'fa-snowflake',
         'monitoring': 'fa-chart-area',
         'its': 'fa-crosshairs',
      };

      const entityTitleMapping = {
         'dns': 'DNS',
         'cert': 'certificate',
         'l3': 'L3 balancer',
         'l7': 'L7 balancer',
         'endpointSet': 'endpoint set',
      };

      const actionTitleMapping = {
         view: key => `View ${entityTitleMapping[key] || key}`,
         delete: key => `Delete ${entityTitleMapping[key] || key}`,
         history: () => 'History',
         edit: key => `Edit ${entityTitleMapping[key] || key}`,
         freeze: key => `Freeze ${entityTitleMapping[key] || key}`,
         monitoring: () => 'Monitoring',
         its: () => 'ITS',
      };

      const controls = controlsMapping[entityKey].map(actionKey => (
         <InfracloudIconButton
            key={`${controlKeyAttr}-${actionKey}`}
            size={'s'}
            onClick={() => {
               this.onIconClick(entityKey, actionKey, ...callbackAttrs);
            }}
            icon={iconClassMapping[actionKey]}
            className={b('table-btn', null, spacings({ 'indent-l': '2xs' }))}
            title={actionTitleMapping[actionKey](entityKey)}
         />
      ));

      return <div className={b('table-controls')}>{controls}</div>;
   }

   renderBackends() {
      const { namespace: { backends, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!backends) {
         return null;
      }

      let hasDeletedItems;

      const items = backends.map(backendData => {
         const {
            backend: {
               dnsRecordStatuses,
               l3Statuses,
               statuses,
               meta: { version, mtime, author, id: backendId },
               spec: { selector: { type } = {}, deleted } = {},
            },
            endpointSet: {
               meta: { version: endpointSetRevision, id: endpointSetId } = {},
               dnsRecordStatuses: dnsRecordStatusesES,
               l3Statuses: l3StatusesES,
               statuses: statusesES,
            } = {},
            backendRevToEndpointSetRevs,
         } = backendData;

         const resolveStatus = get(backendData, 'backend.resolverStatus.lastAttempt.succeeded.status');

         const revisionsData = this.getAllRevisionsStatuses(
            dnsRecordStatuses,
            l3Statuses,
            statuses,
            backendId,
            version,
         );

         const currentBackendRevision = revisionsData.find(x => x.revisionId === version);
         currentBackendRevision.mtime = mtime;
         currentBackendRevision.author = author;
         currentBackendRevision.resolveStatus = resolveStatus;
         currentBackendRevision.deleted = deleted;
         currentBackendRevision.url = `${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/backends/list/${backendId}/show/`;
         hasDeletedItems = hasDeletedItems || deleted;

         if (type !== 'YP_ENDPOINT_SETS_SD') {
            const endpointSetsData = this.getAllRevisionsStatuses(
               dnsRecordStatusesES,
               l3StatusesES,
               statusesES,
               endpointSetId,
            );

            revisionsData.forEach(dataItem => {
               const { revisionId } = dataItem;
               const endpointSetIdsForRevision = backendRevToEndpointSetRevs[revisionId]
                  ? uniq(backendRevToEndpointSetRevs[revisionId].revisions)
                  : [];
               const endpointSetsForRevision = endpointSetsData.filter(es =>
                  endpointSetIdsForRevision.includes(es.revisionId),
               );

               dataItem.endpointSets = endpointSetsForRevision;
            });
         }

         const rows = [];
         revisionsData.forEach((dataItem, index, array) => {
            let versionTitle;
            if (array.length > 1) {
               if (index === 0) {
                  versionTitle = 'current';
               } else {
                  versionTitle = 'old';
               }
            }
            const backendRevisionMarkup = this.createRevisionMarkup(dataItem, versionTitle);

            backendRevisionMarkup.resolveStatus = this.renderResolveStatus(dataItem.resolveStatus);

            if (dataItem.revisionId === version) {
               backendRevisionMarkup.controls = this.renderControls('backend', `${backendId}-${version}`, backendId);
            }

            rows.push(backendRevisionMarkup);

            if (dataItem.endpointSets.length) {
               dataItem.endpointSets.forEach(esData => {
                  const { l7Status, l3Status, dnsStatus, revisionId, revisionCtime } = esData;
                  const isCurrentRevision = revisionId === endpointSetRevision;
                  const controls = isCurrentRevision
                     ? this.renderControls('endpointSet', `${backendId}-${revisionId}-es`, backendId)
                     : null;

                  rows.push({
                     id: <div className={b('endpoint-set-title')}>Endpoint set</div>,
                     l7Status: this.getBackendBalancerStatusMarkup(l7Status, `${revisionId}-l7`),
                     l3Status: this.getBackendBalancerStatusMarkup(l3Status, `${revisionId}-l3`),
                     dnsStatus: this.getBackendBalancerStatusMarkup(dnsStatus, `${revisionId}-dns`),
                     revision: revisionId.split('-').shift(),
                     modified: moment(revisionCtime).fromNow(),
                     controls,
                  });
               });
            }
         });

         return rows;
      });

      const headerItems = [
         { title: 'Backend ID', key: 'id' },
         { title: 'Resolver Status', key: 'resolveStatus' },
         hasDeletedItems ? { title: 'Deleted', key: 'deleted' } : null,
         { title: 'L7 Status', key: 'l7Status' },
         { title: 'L3 Status', key: 'l3Status' },
         { title: 'DNS Status', key: 'dnsStatus' },
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ].filter(x => x);

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('Backends', backends.length)}
            data-test={'namespace-info-backends'}
         >
            {items.length ? (
               <InfracloudTable
                  items={items}
                  headerItems={headerItems}
                  idSelector={({ id, revision }) => `${id}-${revision}`}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('new-backend');
               }, 'Create new')}
               {this.createAddButton(() => {
                  this.onAddClick('existing-backend');
               }, 'Copy backends')}
            </div>
         </InfrauiContainer>
      );
   }

   renderUpstreams() {
      const { namespace: { upstreams, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!upstreams) {
         return null;
      }

      let hasDeletedItems;

      const items = upstreams.map(upstreamData => {
         const {
            statuses,
            meta: { version, mtime, author, id: upstreamId },
            spec: { deleted },
         } = upstreamData;

         const revisionsData = this.getAllRevisionsStatuses([], [], statuses, upstreamId, version);
         const currentBackendRevision = revisionsData.find(x => x.revisionId === version);
         currentBackendRevision.mtime = mtime;
         currentBackendRevision.author = author;
         currentBackendRevision.deleted = deleted;
         currentBackendRevision.url = `${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/upstreams/list/${upstreamId}/show/`;
         hasDeletedItems = hasDeletedItems || deleted;

         const rows = [];
         revisionsData.forEach((dataItem, index, array) => {
            let versionTitle;
            if (array.length > 1) {
               if (index === 0) {
                  versionTitle = 'current';
               } else {
                  versionTitle = 'old';
               }
            }
            const upstreamRevisionMarkup = this.createRevisionMarkup(dataItem, versionTitle);

            if (dataItem.revisionId === version) {
               upstreamRevisionMarkup.controls = this.renderControls(
                  'upstream',
                  `${upstreamId}-${version}`,
                  upstreamId,
               );
            }

            rows.push(upstreamRevisionMarkup);
         });

         return rows;
      });

      const headerItems = [
         { title: 'Upstream ID', key: 'id' },
         { title: 'Status', key: 'l7Status' },
         hasDeletedItems ? { title: 'Deleted', key: 'deleted' } : null,
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ].filter(x => x);

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('Upstreams', upstreams.length)}
            data-test={'namespace-info-upstreams'}
         >
            {items.length ? (
               <InfracloudTable
                  items={items}
                  headerItems={headerItems}
                  idSelector={({ id, revision }) => `${id}-${revision}`}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('new-upstream');
               }, 'Create new')}
               {this.createAddButton(() => {
                  this.onAddClick('existing-upstream');
               }, 'Copy upstreams')}
            </div>
         </InfrauiContainer>
      );
   }

   renderDNSRecords() {
      const { namespace: { dnsRecords, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!dnsRecords) {
         return null;
      }

      const dnsRecordItems = dnsRecords.map(dnsRecord => {
         const {
            meta: { id, version, mtime, author },
            dnsRecordStatus: {
               validated: { status },
            },
         } = dnsRecord;

         const modifiedDate = moment(mtime);
         const unifiedStatus =
            status === 'True'
               ? {
                    status: 'ok',
                    text: 'Validated',
                 }
               : {
                    status: 'error',
                    text: 'Error',
                 };

         const controls = this.renderControls('dns', `${id}-${version}`, id);

         return {
            id: (
               <ExternalLink
                  url={`${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/dns-records/list/${id}/show/`}
                  text={id}
                  hideIcon={true}
               />
            ),
            status: (
               <Status type={'secondary'} state={unifiedStatus.status}>
                  {unifiedStatus.text}
               </Status>
            ),
            unifiedStatus,
            revision: version.split('-').shift(),
            modified: modifiedDate.fromNow(),
            author: (
               <UserAccount
                  uid={author}
                  name={author}
                  avatarId={author}
                  hasAccentLetter={true}
                  url={`https://staff.yandex-team.ru/${author}`}
                  avatarHost={'https://center.yandex-team.ru'}
                  provider={'yandex-team'}
                  cls={b('user-account')}
                  data-test={'namespace-info-dns'}
               />
            ),
            related: 'N/A',
            controls,
         };
      });

      const headerItems = [
         { title: 'Record ID', key: 'id' },
         { title: 'Status', key: 'status' },
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ];

      const total = dnsRecordItems.length;
      const active = dnsRecordItems.filter(x => x.unifiedStatus.status === 'ok').length;
      const error = dnsRecordItems.filter(x => x.unifiedStatus.status === 'error').length;

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('DNS Records', total, active, undefined, error)}
            data-test={'namespace-info-dns'}
         >
            {dnsRecordItems.length ? (
               <InfracloudTable
                  items={dnsRecordItems}
                  headerItems={headerItems}
                  idSelector={'id'}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('dns');
               }, 'Create new')}
            </div>
         </InfrauiContainer>
      );
   }

   renderCertificates() {
      const { namespace: { certificates, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!certificates) {
         return null;
      }

      const items = certificates.map(certificateData => {
         const {
            meta: { id: certificateId, version, mtime, author },
            statuses,
         } = certificateData;

         const revisionsData = this.getAllRevisionsStatuses([], [], statuses, certificateId, version);
         const currentBackendRevision = revisionsData.find(x => x.revisionId === version);
         currentBackendRevision.mtime = mtime;
         currentBackendRevision.author = author;
         currentBackendRevision.url = `${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/certs/list/${certificateId}/show/`;

         const rows = [];
         revisionsData.forEach((dataItem, index, array) => {
            let versionTitle;
            if (array.length > 1) {
               if (index === 0) {
                  versionTitle = 'current';
               } else {
                  versionTitle = 'old';
               }
            }
            const certRevisionMarkup = this.createRevisionMarkup(dataItem, versionTitle);

            if (dataItem.revisionId === version) {
               const controls = this.renderControls('cert', `${certificateId}-${version}`, certificateId);

               certRevisionMarkup.controls = controls;
            }

            rows.push(certRevisionMarkup);
         });

         return rows;
      });

      const headerItems = [
         { title: 'Certificate ID', key: 'id' },
         { title: 'Status', key: 'l7Status' },
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ];

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('Certificates', certificates.length)}
            data-test={'namespace-info-certs'}
         >
            {items.length ? (
               <InfracloudTable
                  items={items}
                  headerItems={headerItems}
                  idSelector={({ id, revision }) => `${id}-${revision}`}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('cert');
               }, 'Order new')}
            </div>
         </InfrauiContainer>
      );
   }

   renderContent() {
      return (
         <>
            {this.renderOwners()}
            {this.renderL3Balancers()}
            {this.renderBalancers()}
            {this.renderUpstreams()}
            {this.renderBackends()}
            {this.renderDNSRecords()}
            {this.renderCertificates()}
         </>
      );
   }

   renderCreationMessage() {
      const {
         updateTime,
         namespace: {
            meta: { id },
         },
      } = this.props;
      const { namespace } = updateTime;

      return (
         <div className={b('creation-message', null, spacings({ 'indent-t': '5xl' }))}>
            <p>This namespace is being created.</p>
            <p>This page will refresh itself every 5 seconds to keep you informed.</p>
            <p>
               For more information <ExternalLink url={EXTERNAL_LINKS.awacsNamespace(id)} text={'go to AWACS'} />
            </p>
            <p className={b('updated')}>Updated at: {moment(namespace).format('HH:mm:ss')}</p>
         </div>
      );
   }

   renderSpiner() {
      return (
         <div className={b('spiner', null, spacings({ 'indent-t': '5xl' }))}>
            <Spin size={'l'} progress />
         </div>
      );
   }

   renderBreadcrumbs() {
      const {
         match: {
            params: { namespaceId },
         },
      } = this.props;
      return (
         <Breadcrumbs
            className={b('breadcrumbs', null, spacings({ 'indent-b': 'xl' }))}
            links={[
               { name: 'All namespaces', url: createHref(routes.balancers) },
               { name: namespaceId, url: createHref(routes.namespaceInfo, { namespaceId }) },
            ]}
         />
      );
   }

   renderHeader() {
      const {
         match: {
            params: { namespaceId },
         },
      } = this.props;

      return (
         <>
            {this.renderBreadcrumbs()}
            <div className={b('header')} data-test={'namespace-info-header'}>
               <div className={b('header-title', null)}>{namespaceId}</div>
               {/* <span className={b('fetch-controls', null, spacings({'indent-l': 's'}))}>
                    <span className={b('fetch-text')}>Auto-updating of data is disabled</span>
                    <InfracloudIconButton
                        icon="fa-sync"
                        size="m"
                        className={b('icon', {header: true}, spacings({'indent-l': 's'}))}
                        onClick={this.updateNamespace}
                        title="Update namespace state"
                    />
                </span> */}
               <InfracloudIconButton
                  icon={'fa-pencil'}
                  size={'m'}
                  className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                  onClick={() => this.onIconClick('namespace', 'view')}
                  title={'Edit'}
               />
               <div style={{ marginLeft: '8px' }}>
                  <ButtonLink to={urlBuilder.awacsNamespaceMap(namespaceId)}>
                     Map <span style={{ color: 'red', fontSize: '0.9em' }}>new!</span>
                  </ButtonLink>
               </div>

               <span className={b('header-controls')}>
                  <InfracloudIconButton
                     icon={'fa-chart-area'}
                     size={'m'}
                     className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                     onClick={() => this.onIconClick('namespace', 'monitoring')}
                     title={'Monitoring'}
                  />
                  <InfracloudIconButton
                     icon={'fa-project-diagram'}
                     size={'m'}
                     className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                     onClick={() => this.onIconClick('namespace', 'map')}
                     title={'Map'}
                  />
                  <InfracloudIconButton
                     icon={'fa-bell'}
                     size={'m'}
                     className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                     onClick={() => this.onIconClick('namespace', 'alert')}
                     title={'Alerting (beta)'}
                  />
                  <InfracloudIconButton
                     icon={'fa-crosshairs'}
                     size={'m'}
                     className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                     onClick={() => this.onIconClick('namespace', 'its')}
                     title={'ITS'}
                  />
                  <InfracloudIconButton
                     icon={'fa-trash-alt'}
                     size={'m'}
                     className={b('icon', { header: true }, spacings({ 'indent-l': 's' }))}
                     onClick={() => this.onIconClick('namespace', 'delete')}
                     title={'Delete namespace'}
                  />
               </span>
            </div>
         </>
      );
   }

   renderOwners() {
      const { namespace, abcServices, staffGroups } = this.props;
      if (!namespace) {
         return null;
      }

      const {
         meta: {
            abcServiceId,

            auth: { staff: { owners: { logins = [], groupIds = [] } = {} } = {} },
         },
      } = namespace;

      const abcService = abcServices[abcServiceId.toString()];
      const ownerService = abcService
         ? {
              id: abcService.id,
              title: abcService.name,
           }
         : null;

      const ownerGroups = groupIds.map(x => Number(x));
      const groups = staffGroups.filter(group => ownerGroups.includes(group.id)).map(group => group.name);

      return (
         <InfracloudOwners abcService={ownerService} owners={logins} groups={groups} data-test={'balancer-owners'} />
      );
   }

   renderBalancers() {
      const { namespace: { balancers, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!balancers) {
         return null;
      }

      const balancerItems = balancers.map(balancer => {
         const {
            meta: { id, version, mtime, author },
            status,
         } = balancer;
         const serviceId = get(balancer, 'spec.configTransport.nannyStaticFile.serviceId');

         const modifiedDate = moment(mtime);
         const unitedStatus = getUnifiedStatus(status);

         const controls = this.renderControls('l7', `${id}-${version}`, id);

         return {
            id: (
               <ExternalLink
                  url={`${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/balancers/list/${id}/show/`}
                  text={id}
                  hideIcon={true}
               />
            ),
            status: (
               <Status type={'secondary'} state={unitedStatus.status}>
                  {unitedStatus.text}
               </Status>
            ),
            unitedStatus,
            revision: version.split('-').shift(),
            modified: modifiedDate.fromNow(),
            author: (
               <UserAccount
                  uid={author}
                  name={author}
                  avatarId={author}
                  hasAccentLetter={true}
                  url={`https://staff.yandex-team.ru/${author}`}
                  avatarHost={'https://center.yandex-team.ru'}
                  provider={'yandex-team'}
                  cls={b('user-account')}
               />
            ),
            service: serviceId ? (
               <ExternalLink text={serviceId} url={`${NANNY_UI_DOMAIN}/services/catalog/${serviceId}/`} />
            ) : null,
            related: 'N/A',
            controls,
         };
      });

      const headerItems = [
         { title: 'Balancer ID', key: 'id' },
         { title: 'Status', key: 'status' },
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         { title: 'Service ID', key: 'service' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ];

      const total = balancers.length;
      const active = balancerItems.filter(x => x.unitedStatus.status === 'ok').length;
      const progress = balancerItems.filter(x => x.unitedStatus.status === 'progress').length;
      const error = balancerItems.filter(x => x.unitedStatus.status === 'error').length;

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('L7 Balancers', total, active, progress, error)}
            data-test={'namespace-info-l7'}
         >
            {balancerItems.length ? (
               <InfracloudTable
                  items={balancerItems}
                  headerItems={headerItems}
                  idSelector={'id'}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('new-L7');
               }, 'Create new')}
               {this.createAddButton(() => {
                  this.onAddClick('existing-L7');
               }, 'Create new from Nanny-service')}
            </div>
         </InfrauiContainer>
      );
   }

   renderL3Balancers() {
      const { namespace: { l3Balancers, meta: { id: namespaceId } = {} } = {} } = this.props;

      if (!l3Balancers) {
         return null;
      }

      const balancerItems = l3Balancers.map(balancer => {
         const {
            meta: { id, version, mtime, author },
            spec: { l3mgrServiceId } = {},
            l3Status,
         } = balancer;

         const modifiedDate = moment(mtime);
         const unitedStatus = getUnifiedStatus(l3Status);

         const controls = this.renderControls('l3', `${id}-${version}`, id);

         return {
            id: (
               <ExternalLink
                  url={`${NANNY_UI_DOMAIN}/awacs/namespaces/list/${namespaceId}/l3-balancers/list/${id}/show/`}
                  text={id}
                  hideIcon={true}
               />
            ),
            status: (
               <Status type={'secondary'} state={unitedStatus.status}>
                  {unitedStatus.text}
               </Status>
            ),
            unitedStatus,
            revision: version.split('-').shift(),
            modified: modifiedDate.fromNow(),
            author: (
               <UserAccount
                  uid={author}
                  name={author}
                  avatarId={author}
                  hasAccentLetter={true}
                  url={`https://staff.yandex-team.ru/${author}`}
                  avatarHost={'https://center.yandex-team.ru'}
                  provider={'yandex-team'}
                  cls={b('user-account')}
               />
            ),
            l3mgrService: (
               <ExternalLink text={l3mgrServiceId} url={`https://l3.tt.yandex-team.ru/service/${l3mgrServiceId}`} />
            ),
            related: 'N/A',
            controls,
         };
      });

      const headerItems = [
         { title: 'Balancer ID', key: 'id' },
         { title: 'Status', key: 'status' },
         { title: 'Revision', key: 'revision' },
         { title: 'Last changes', key: 'modified' },
         { title: 'Author', key: 'author' },
         { title: 'Service ID', key: 'l3mgrService' },
         // {title: 'Related', key: 'related'},
         { title: '', key: 'controls' },
      ];

      const total = l3Balancers.length;
      const active = balancerItems.filter(x => x.unitedStatus.status === 'ok').length;
      const progress = balancerItems.filter(x => x.unitedStatus.status === 'progress').length;
      const error = balancerItems.filter(x => x.unitedStatus.status === 'error').length;

      return (
         <InfrauiContainer
            spacings={{ 'indent-t': '3xl' }}
            headerSpacings={{ 'indent-b': 'l' }}
            header={this.renderSectionTitle('L3 Balancers', total, active, progress, error)}
            data-test={'namespace-info-l3'}
         >
            {balancerItems.length ? (
               <InfracloudTable
                  items={balancerItems}
                  headerItems={headerItems}
                  idSelector={'id'}
                  itemSpacing={{ 'space-l': 's', 'space-r': 'l', 'space-v': 'xs' }}
               />
            ) : null}
            <div className={b('section-controls', null, spacings({ 'indent-t': 's' }))}>
               {this.createAddButton(() => {
                  this.onAddClick('new-L3');
               }, 'Create new')}
               {this.createAddButton(() => {
                  this.onAddClick('existing-L3');
               }, 'Add existing')}
            </div>
         </InfrauiContainer>
      );
   }

   render() {
      const { namespace, loadingFlags } = this.props;

      let content;
      if (namespace) {
         const status = get(namespace, 'order.status.status');
         const isReady = !namespace.order || ['FINISHED', 'CANCELLED'].includes(status);

         if (isReady) {
            content = this.renderContent();
         } else {
            content = this.renderCreationMessage();
         }
      } else if (loadingFlags.namespace) {
         content = this.renderSpiner();
      } else {
         content = <EmptyContainer type={EmptyContainerType.NotFound} title={'Namespace does not exist'} />;
      }

      return (
         <PageWrapper title={this.props.namespace ? `Balancer: ${this.props.namespace.meta.id}` : undefined}>
            <Page>
               <InfrauiContainer
                  header={namespace && this.renderHeader()}
                  spacings={{ 'space-h': '2xl', 'indent-t': 'xl' }}
                  headerSpacings={{ 'indent-b': '3xl' }}
                  contentSpacings={{ 'space-b': '3xl' }}
               >
                  {content}
               </InfrauiContainer>
            </Page>
         </PageWrapper>
      );
   }
}

const mapStateToProps = state => ({
   namespace: state.balancers.namespace,
   abcServices: state.abc.services,
   staffGroups: state.staff.groups,
   updateTime: state.balancers.updateTime,
   loadingFlags: state.balancers.loadingFlags,
});

export default compose(withRouter, withNotifications, connect(mapStateToProps))(NamespaceInfoPage);
