import { Button } from '@yandex-cloud/uikit';
import { EmptyContainer, EmptyContainerType } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import dateFormat from 'dateformat';
import { Link, Spin } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { compose } from 'recompose';
import { createStructuredSelector } from 'reselect';

import { YpErrorTooltip } from '../../../components/network';
import { urlBuilder } from '../../../models';
import {
   createProject,
   fetchAvailableObjects,
   fetchProjects,
   fetchStages,
   getStages,
   selectAvailableObjects,
   selectContinuationTokens,
   selectNetworkErrorsByKeys,
   selectProjects,
   selectStageNamesByProject,
   selectStages,
} from '../../../redux';
import { ApiServicesContext } from '../../../services';
import { setUrlQuery, decodeYPErrorMessage, decodeYPErrorMessages } from '../../../utils';
import { Breadcrumbs } from '../../components/Breadcrumbs/Breadcrumbs';
import withNotifications from '../../components/hoc/withNotifications';
import Page from '../../components/Page/Page';
import PageWrapper from '../../components/PageWrapper/PageWrapper';
import RouteLink from '../../components/RouteLink/RouteLink';
import Tooltip from '../../components/Tooltip/Tooltip';
import { isMatched, parseQueryWithTags } from '../../utils/queryWithTags';

import ProjectCardContainer from '../ProjectCardContainer/ProjectCardContainer';
import ProjectCreateForm from '../ProjectCreateForm/ProjectCreateForm';
import { PageMainFilters } from './PageMainFilters';
import { PageMainCreateProjectError } from '../../../pages/main/components';

import './PageMain.scss';

const { login } = window.USER;

const b = block('page-main');

const requestKeyNamespace = 'PageMain';
const pageMainRequestKeys = {
   all: `${requestKeyNamespace}-all`,
   myProjects: `${requestKeyNamespace}-myProjects`,
   myStages: `${requestKeyNamespace}-myStages`,
   createProject: `${requestKeyNamespace}-createProject`,
};

class PageMain extends Component {
   static propTypes = {
      match: PropTypes.object.isRequired,
      history: PropTypes.object.isRequired,
      notifications: PropTypes.object.isRequired,
      projects: PropTypes.object.isRequired,
      stages: PropTypes.object.isRequired,
   };

   static contextType = ApiServicesContext;

   state = {
      sorts: {
         orderBy: 'desc',
         nameBy: 'created',
      },
      filters: {
         my: 'yes',
         project: '',
         stage: '',
         objectType: 'project',
      },
      isLoading: false,
      projectsOpened: {},
      search: {
         caption: '',
      },
      searchCache: {},
      opened: {},
   };

   searchCache = {};

   fetchedObjects = {
      my: false,
      all: false,
   };

   componentDidMount() {
      this.asyncSetState = stateParam =>
         new Promise(resolve => {
            this.setState(stateParam, () => resolve());
         });

      const { notifications } = this.props;
      const search = new URLSearchParams(this.props.location.search);

      this.asyncSetState(state => ({
         isLoading: true,
         filters: {
            ...state.filters,
            my: search.get('my') === 'no' ? 'no' : 'yes',
            objectType: search.get('project') ? 'project' : 'stage',
            project: search.get('project') || '',
            stage: search.get('stage') || '',
            name: search.get('project') || search.get('stage') || '',
         },
      }))
         .then(() => this.setFilters({ reset: true }))
         .then(() =>
            this.asyncSetState({
               isLoading: false,
            }),
         )
         .then(() => {
            this.fetchAllProjects();
         })
         .catch(err => {
            console.log(err);
            let errorMessage = '';

            if (err.response?.data) {
               errorMessage = err.response.data.message || err.response.data;
            }

            notifications.add({
               level: 'error',
               title: `${err.message}`,
               message: err.response?.data?.ytError?.message
                  ? decodeYPErrorMessage(err.response.data.ytError.message)
                  : decodeYPErrorMessage(errorMessage),
               ytError: decodeYPErrorMessages(err.response?.data?.ytError),
               autoDismiss: 0,
            });
            this.setState({
               isLoading: false,
            });
         });
   }

   onChangeFilterMy(value) {
      this.setState(
         state => ({
            filters: {
               ...state.filters,
               my: value,
            },
         }),
         () => this.setFilters(),
      );
   }

   onChangeFilterObjectType(value) {
      this.setState(
         state => ({
            filters: {
               ...state.filters,
               objectType: value,
               stage: '',
               project: '',
               [value]: state.filters.name,
            },
         }),
         () => {
            this.setFilters();
         },
      );
   }

   onSearchButtonClick() {
      this.setFilters();
   }

   onChangeFilterName(name) {
      this.setState(state => ({
         filters: {
            ...state.filters,
            name,
         },
      }));
   }

   onFiltersReset() {
      this.setState(
         {
            filters: {
               my: 'yes',
               project: '',
               stage: '',
               name: '',
               objectType: 'stage',
            },
            opened: {},
         },
         () => this.setFilters(),
      );
   }

   setFilters({ reset } = {}) {
      const { fetchFilteredProjects, objectIdsWithLogins } = this.props;
      const {
         filters: { name: query, objectType, my },
      } = this.state;
      const getLoadDataPromise = () => {
         if (my === 'yes' && !this.fetchedObjects.my) {
            this.fetchedObjects.my = true;
            return this.fetchMyObjects({ reset });
         }
         if (my === 'no' && !this.fetchedObjects.all) {
            this.fetchedObjects.all = true;
            return this.fetchStartAllObjects({ reset });
         }
         return Promise.resolve();
      };

      const parsedQuery = parseQueryWithTags(query);
      const name = parsedQuery.names.join('');
      const { tags } = parsedQuery;

      const searchKey = `${objectType}/${query}/${my}`;
      const getFilterPromise = () => {
         const lastSearchTimeout = this.searchCache[searchKey];
         const searchAvailable = !lastSearchTimeout || new Date() - lastSearchTimeout > 10000; // 10 sec
         if (query && searchAvailable) {
            this.searchCache[searchKey] = new Date();

            return this.asyncSetState({ isLoading: true }).then(() => {
               if (my === 'yes') {
                  const stageIds = Object.keys(objectIdsWithLogins.stage || {});
                  const myStageIds = stageIds.filter(
                     stageId => ((objectIdsWithLogins.stage || {})[stageId] || {})[login],
                  );
                  const myFilteredStageIds = myStageIds.filter(stageId => isMatched(parsedQuery, stageId, null));
                  return this.props.getStages({
                     objectIds: myFilteredStageIds,
                     paths: ['/labels', '/meta', '/spec', '/status'],
                     fetchTimestamps: true,
                  });
               }
               if (objectType === 'project') {
                  return fetchFilteredProjects({
                     limit: 20,
                     substring: name,
                     tags,
                     paths: p => [p.meta.id, p.labels.tags],
                  });
               }
               return this.props.fetchStages({
                  limit: 100,
                  substring: name,
                  tags,
               });
            });
         }
         return Promise.resolve();
      };
      return this.asyncSetState(state => ({
         filters: {
            ...state.filters,
            [objectType]: name,
         },
      }))
         .then(() => this.setURLParams({ my, objectType, name: query }))
         .then(() => getLoadDataPromise())
         .then(() => getFilterPromise())
         .then(() => {
            const opened = {};
            if (query) {
               for (const projectName of this.getProjectList()) {
                  opened[projectName] = objectType === 'stage';
               }
            }
            this.setState(state => ({
               isLoading: false,
               opened: {
                  ...state.opened,
                  ...opened,
               },
            }));
         });
   }

   setURLParams({ my, objectType, name }) {
      const params = {
         my,
         [objectType]: name,
      };

      setUrlQuery(this.props.history, this.props.location, params, false);
   }

   getProjectList() {
      const { projects, stageNamesByProject, objectIdsWithLogins, stages } = this.props;
      const {
         filters: { project, stage, name, objectType, my },
      } = this.state;
      const isMy = list => list.some(stageId => ((objectIdsWithLogins.stage || {})[stageId] || {})[login]);
      return [...new Set([...Object.keys(stageNamesByProject), ...Object.keys(projects)])]
         .filter(projectName => {
            if (my === 'yes') {
               const stageIds = Object.keys(stageNamesByProject[projectName] || {}) || [];
               return isMy(stageIds) || ((objectIdsWithLogins.project || {})[projectName] || {})[login];
            }
            return true;
         })
         .filter(projectName => {
            if (objectType === 'project') {
               const projectItem = projects[projectName];

               return isMatched(name || project, projectName, projectItem ? projectItem.labels?.tags ?? [] : null);
            }

            const query = name || stage;
            if (!query) {
               return true;
            }

            const stageIds = Object.keys(stageNamesByProject[projectName] || {}) || [];
            return stageIds.some(stageId => {
               const stageTags = stages[stageId]?.labels?.tags ?? [];

               return isMatched(query, stageId, stageTags);
            });
         })
         .sort((a, b_) => (a.toLowerCase() > b_.toLowerCase() ? 1 : -1));
   }

   fetchMyObjects({ reset } = {}) {
      const { fetchMyObjectIds, getStages: asyncGetStages } = this.props;
      return fetchMyObjectIds({
         meta: {
            project: reset,
         },
         requestKey: pageMainRequestKeys.myProjects,
         requests: [
            {
               objectType: 'project',
               login,
               limit: 100000,
            },
         ],
      })
         .then(() =>
            fetchMyObjectIds({
               meta: {
                  stage: reset,
               },
               requestKey: pageMainRequestKeys.myStages,
               requests: [
                  {
                     objectType: 'stage',
                     login,
                     limit: 100000,
                  },
               ],
            }),
         )
         .then(() => {
            const { objectIdsWithLogins } = this.props;
            const stages = objectIdsWithLogins.stage || {};
            const myStageIds = Object.keys(stages).filter(id => stages[id][login]);
            const limit = 100; // количество стейджей в запросе
            const parts = [];
            for (let i = 0; i < myStageIds.length; i += 1) {
               const j = (i - (i % limit)) / limit;
               if (!parts[j]) {
                  parts[j] = [];
               }
               parts[j].push(myStageIds[i]);
            }
            return Promise.allSettled(
               parts.map(ids =>
                  asyncGetStages({
                     objectIds: ids,
                     paths: ['/labels', '/meta', '/spec', '/status'],
                     fetchTimestamps: true,
                  }),
               ),
            );
         });
   }

   fetchStartAllObjects({ reset } = {}) {
      const { fetchFilteredProjects } = this.props;
      return fetchFilteredProjects({
         meta: { reset },
         limit: 10,
         substring: '',
         reset,
         paths: p => [p.meta.id, p.labels.tags],
      });
   }

   fetchAllProjects({ reset } = {}) {
      const { fetchFilteredProjects } = this.props;
      return fetchFilteredProjects({
         requestKey: pageMainRequestKeys.all,
         meta: { reset },
         limit: 100000,
         loadAll: true,
         paths: p => [p.meta.id, p.labels.tags],
      });
   }

   createProject({ values }) {
      const { createProject: asyncCreateProject } = this.props;
      asyncCreateProject({
         meta: {
            id: values.project_name,
            acl: [
               {
                  action: 'allow',
                  permissions: ['read', 'write', 'create', 'ssh_access', 'root_ssh_access', 'read_secrets'],
                  subjects: window.CONFIG.deployRobots ?? [],
               },
            ],
         },
         spec: {
            // eslint-disable-next-line no-underscore-dangle
            account_id: values.temporary_account ? 'tmp' : `abc:service:${values.abc._id}`,
         },
         requestKey: pageMainRequestKeys.createProject,
      })
         .then(reduxEvent => {
            // TODO: при переписывании на функциональные компоненты и хуки честно разделить ошибки от удачных ответов
            if (reduxEvent.type.endsWith('rejected')) {
               throw reduxEvent.payload.error ?? new Error(reduxEvent.error.message ?? 'Error creating project');
            }
            this.props.history.push(urlBuilder.project(values.project_name));
         })
         .catch(() => {
            this.setState({
               isSubmittingProjectForm: false,
               newProjectIsSubmitting: false,
            });
         });
   }

   renderProjectCreated(project) {
      let projectCreated = '';
      try {
         projectCreated = dateFormat(new Date(project.meta.creation_time / 1000), 'H:MM, mmm d, yyyy');
      } catch (err) {
         console.log(err);
      }
      return projectCreated;
   }

   renderProjectLink(stage) {
      return (
         <RouteLink to={urlBuilder.stage(stage.meta.id)} theme={'normal'}>
            {stage.meta.id}
         </RouteLink>
      );
   }

   renderProjectName(project) {
      return (
         <div className={b('project-name')} data-test={`${project.meta.id}`}>
            {this.renderProjectLink(project)}
         </div>
      );
   }

   renderFilters() {
      const { filters } = this.state;

      return <PageMainFilters filters={filters} setFilters={f => this.setState({ filters: f }, this.setFilters)} />;
   }

   renderProjectList() {
      const { filters, opened } = this.state;
      const projectList = this.getProjectList();

      return projectList.map((projectName, i) => {
         if (projectName === '' && filters.my === 'yes') {
            return null;
         }

         return (
            <ProjectCardContainer
               key={projectName}
               projectName={projectName}
               opened={opened[projectName]}
               startOpened={opened[projectName] && !opened[projectList[i - 1]]}
               endOpened={opened[projectName] && !opened[projectList[i + 1]]}
               toggleCard={() => {
                  this.setState(state => ({
                     opened: {
                        ...state.opened,
                        [projectName]: !state.opened[projectName],
                     },
                  }));
               }}
               filters={filters}
            />
         );
      });
   }

   renderContent() {
      const {
         stages,
         stageNamesByProject,
         continuationTokens,
         fetchFilteredProjects,
         projects,
         networkErrors,
      } = this.props;
      const {
         isLoading,
         filters: { project: name, my, objectType },
      } = this.state;
      const stagesByProject = {};
      for (const projectName of Object.keys(stageNamesByProject)) {
         const names = Object.keys(stageNamesByProject[projectName]);
         stagesByProject[projectName] = names.map(e => stages[e]);
      }
      const projectList = this.getProjectList();
      return (
         <div>
            {(() => {
               if (isLoading) {
                  return (
                     <div className={b('table-spinner')}>
                        <Spin size={'l'} progress />
                     </div>
                  );
               }
               if (projectList.length > 0) {
                  return (
                     <>
                        <div className={b('project-list')}>{this.renderProjectList()}</div>
                        {my === 'no' &&
                           !((continuationTokens.projects[name] || {})[Symbol.for('min')] || {}).terminator && (
                              <div style={{ margin: '16px 32px' }}>
                                 <Link
                                    theme={'pseudo'}
                                    onClick={() => {
                                       fetchFilteredProjects({
                                          limit: 10000,
                                          continuationToken: (
                                             (continuationTokens.projects[name] || {})[Symbol.for('min')] || {}
                                          ).token,
                                       });
                                    }}
                                 >
                                    Load More Projects
                                 </Link>
                              </div>
                           )}
                     </>
                  );
               }
               if (Object.keys(stageNamesByProject).length > 0 || Object.keys(projects).length > 0) {
                  return (
                     <EmptyContainer
                        type={EmptyContainerType.NotFound}
                        title={`No ${objectType}s matching your criteria`}
                        description={'Try to change a search query or filters'}
                        className={b('empty-container')}
                     />
                  );
               }
               const existErrors = Object.values(networkErrors).some(Boolean);
               return (
                  <>
                     <EmptyContainer
                        type={existErrors ? EmptyContainerType.Error : EmptyContainerType.EmptyState}
                        title={existErrors ? 'Api Error' : 'Welcome to Yandex Deploy'}
                        description={
                           existErrors ? (
                              <div className={b('errors')}>
                                 {Object.values(networkErrors)
                                    .filter(Boolean)
                                    .map(item => (
                                       <YpErrorTooltip error={item.error} request={item.request} />
                                    ))}
                              </div>
                           ) : (
                              <p>You haven't launched any projects yet</p>
                           )
                        }
                        className={b('empty-container')}
                     />
                  </>
               );
            })()}
         </div>
      );
   }

   renderBreadcrumbs() {
      return (
         <div className={b('breadcrumbs')}>
            <Breadcrumbs links={[{ name: 'All projects', url: urlBuilder.projects() }]} />
         </div>
      );
   }

   renderNewProjectForm() {
      return (
         <div className={b('create-project')}>
            <ProjectCreateForm
               submitText={'Create'}
               onSubmit={values => {
                  this.setState(
                     {
                        newProjectIsSubmitting: true,
                     },
                     () => this.createProject({ values }),
                  );
               }}
               handleCancel={() => {
                  this.setState({
                     newProjectFormVisible: false,
                  });
               }}
               initialValues={{
                  abc: '',
                  project_name: '',
               }}
               projects={this.props.projects}
               isSubmitting={this.state.newProjectIsSubmitting || false}
            />
            <PageMainCreateProjectError className={b('network-error')} requestKey={pageMainRequestKeys.createProject} />
         </div>
      );
   }

   renderCreateProjectButton() {
      return (
         <div style={{ margin: '8px', marginRight: '32px' }} data-test={'create-project-button'}>
            <Tooltip
               text={this.renderNewProjectForm()}
               visible={this.state.newProjectFormVisible}
               clickable={true}
               modal={true}
               mix={'create-project'}
            >
               <Button
                  onClick={() => {
                     this.setState({
                        newProjectFormVisible: true,
                     });
                  }}
                  view={'action'}
               >
                  Create new project
               </Button>
            </Tooltip>
         </div>
      );
   }

   renderHeaderExtra() {
      return (
         <div style={{ display: 'grid', gridTemplateColumns: '1fr auto' }}>
            {this.renderFilters()}
            {this.renderCreateProjectButton()}
         </div>
      );
   }

   render() {
      const title = null;

      return (
         <PageWrapper title={'Projects'}>
            <Page title={title} className={b()} dataTest={'page-main'} headerExtra={this.renderHeaderExtra()}>
               {this.renderContent()}
            </Page>
         </PageWrapper>
      );
   }
}

const mapStateToProps = createStructuredSelector({
   stageNamesByProject: selectStageNamesByProject,
   stages: selectStages,
   projects: selectProjects,
   continuationTokens: selectContinuationTokens,
   objectIdsWithLogins: selectAvailableObjects,
   networkErrors: state => selectNetworkErrorsByKeys(state, Object.values(pageMainRequestKeys)),
});

const mapDispatchToProps = {
   fetchFilteredProjects: ({ meta, requestKey, ...params }) =>
      fetchProjects.withRequestKey(requestKey).withMeta(meta)({ ...params }),
   fetchStages,
   fetchMyObjectIds: ({ meta, requestKey, ...params }) =>
      fetchAvailableObjects.withRequestKey(requestKey).withMeta(meta)({ ...params }),
   getStages: ({ meta, requestKey, ...params }) => getStages.withRequestKey(requestKey).withMeta(meta)({ ...params }),
   createProject: ({ requestMeta, requestKey, ...params }) =>
      createProject.withRequestKey(requestKey).withMeta(requestMeta)({ ...params }),
};

export default compose(withNotifications, connect(mapStateToProps, mapDispatchToProps))(PageMain);
