import '@fortawesome/fontawesome-pro/css/light.min.css';
import '@fortawesome/fontawesome-pro/css/solid.min.css';
import { HelpPopover } from '@yandex-cloud/uikit';
import {
   ButtonLink,
   EmptyContainer,
   EmptyContainerType,
   ExternalLink,
   Hint,
   isEqual,
   WarningPlate,
} from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import dateFormat from 'dateformat';
import { Button, Spin } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component, Fragment } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';
import { createStructuredSelector } from 'reselect';
import { firstValueFrom } from 'rxjs';
import { GroupLink, ManageRoleLink, RolesList } from '../../../components';
import { setToListOptions, TagList } from '../../../components/forms/inputs/TagList/TagList';
import { UserLink } from '../../../components/lib';
import { TagsHint } from '../../../components/stage-levels/stage/hints';
import { EXTERNAL_LINKS, urlBuilder } from '../../../models';
import { rolesData } from '../../../models/roles';
import {
   deleteProject,
   fetchStages,
   getProjects,
   projectsSlice,
   selectContinuationTokens,
   selectProjects,
   selectStageNamesByProject,
   selectStages,
} from '../../../redux';

import { ApiServicesContext, ypApi } from '../../../services';
import { getConfig } from '../../../services/Config';
import { Breadcrumbs } from '../../components/Breadcrumbs/Breadcrumbs';
import ColoredInput from '../../components/ColoredInput/ColoredInput';
import ConfirmDialog from '../../components/ConfirmDialog/ConfirmDialog';
import withNotifications from '../../components/hoc/withNotifications';
import Note from '../../components/Note/Note';
import Page from '../../components/Page/Page';
import PageWrapper from '../../components/PageWrapper/PageWrapper';
import RouteLink from '../../components/RouteLink/RouteLink';
import Select from '../../components/Select/Select';
import SideBar from '../../components/SideBar/SideBar';
import StageRolesCard from '../../components/StageRolesCard/StageRolesCard';
import Tooltip from '../../components/Tooltip/Tooltip';
import { fetchAbcService, getAbcServices } from '../../store/reducers/abc';
import { getClusters } from '../../store/reducers/clusters';
import { getProjectAcl, selectProjectsAcl } from '../../store/reducers/deployAuth';
import { fetchGroups, selectFetchedGroups, selectGroups } from '../../store/reducers/staff';

import { decodeYPErrorMessage, decodeYPErrorMessages } from '../../../utils';
import ProjectEditForm from '../ProjectEditForm/ProjectEditForm';

import './Project.scss';

const b = block('project');

const projectAclGroups = rolesData.rolesByObject.project.map(role => role.id);
const stageAclGroups = rolesData.rolesByObject.stage.map(role => role.id);
const boxTypeAclGroups = rolesData.rolesByObject.box.map(role => role.id);

class Project extends Component {
   static propTypes = {
      match: PropTypes.object.isRequired,
      history: PropTypes.object.isRequired,
      project: PropTypes.object,
      projects: PropTypes.object,
      defaultOS: PropTypes.array,
      location: PropTypes.object,
      notifications: PropTypes.object.isRequired,
      getProjects: PropTypes.func.isRequired,
      getProjectAcl: PropTypes.func.isRequired,
      stageNamesByProject: PropTypes.object,
      projectsAcl: PropTypes.object,
   };

   static contextType = ApiServicesContext;

   state = {
      confirmDialog: null,
      isProjectLoaded: false,
      isEditMode: false,
      isSubmitting: false,
      isLoading: true,
      currentStageId: '',
      inherited: {},
      searchStage: '',
      stagesLoading: false,
   };

   fetchedAbc = {};

   actions = {
      edit: {
         icon: 'pen',
         action: () => this.setState(s => ({ isEditMode: !s.isEditMode })),
         isActive: () => this.state.isEditMode,
      },
      copy: {
         icon: 'copy',
         action: () => {},
      },
      delete: {
         icon: 'trash-alt',
         action: this.onDeleteProject.bind(this),
      },
   };

   // eslint-disable-next-line react/sort-comp
   get projectId() {
      return this.props.match.params.projectId;
   }

   componentDidMount() {
      this.fetchData({ init: true });
   }

   componentWillUnmount() {
      window.clearTimeout(this.fetchDataTimeout);
      this.fetchDataTimeout = null;
   }

   onDeleteProject() {
      const { stageNamesByProject } = this.props;
      const stageIds = Object.keys(stageNamesByProject[this.projectId] || {});
      if (stageIds.length > 0) {
         this.openConfirmDialog({
            title: 'Project deletion',
            message: `The project has ${stageIds.length} stages. Remove them first`,
            actionText: 'OK',
            actionTheme: 'action',
            actionHandler: this.closeConfirmDialog.bind(this),
            cancelText: 'Cancel',
         });
      } else {
         this.openConfirmDialog({
            title: 'Project deletion',
            message: `Caution! You are about to delete a project ${this.projectId}.`,
            actionText: 'Delete',
            actionTheme: 'action',
            actionHandler: this.deleteProject.bind(this),
            confirmation: `Yes, I know what I'm doing, delete ${this.projectId}`,
            cancelText: 'Cancel',
         });
      }
   }

   onSaveProjectEdit(values) {
      const { notifications } = this.props;
      const { projects } = this.props;
      const project = projects[this.projectId];
      /* https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9 */
      // const boxTypes = values.box_types ? values.box_types.split(',') : [];
      // const newBoxTypes = values.new_box_types ? values.new_box_types.split(',') : [];
      // const deleteBoxTypes = values.delete_box_types ? values.delete_box_types.split(',') : [];
      // const list = [...boxTypes, ...newBoxTypes];
      // const filteredList = list.filter(type => !deleteBoxTypes.includes(type));
      const updateTags = !isEqual(project?.labels?.tags, values.tags);

      const paths = {};

      // https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9
      // paths['/spec/user_specific_box_types']=  filteredList.length > 0 ? filteredList : [];

      // eslint-disable-next-line no-underscore-dangle
      paths['/spec/account_id'] = values.temporary_account ? 'tmp' : `abc:service:${values.abc._id}`;

      if (updateTags) {
         paths['/labels/tags'] = values.tags;
      }

      if (values.monitoring_project?.projectId !== project?.spec?.monitoring_project) {
         paths['/spec/monitoring_project'] = values.monitoring_project?.projectId;
      }

      firstValueFrom(
         ypApi.updateProject({
            id: this.projectId,
            paths,
         }),
      )
         .then(() => {
            this.fetchProject();
            this.setState({
               boxTypesEdit: false,
               editProjectIsSubmitting: false,
               isEditMode: false,
            });
         })
         .catch(err => {
            console.error(err);
            let errorMessage = '';

            if (err?.data) {
               errorMessage = err.data.message || '';
            }

            notifications.add({
               level: 'error',
               title: `Update project error`,
               message: err?.data?.ytMessage
                  ? decodeYPErrorMessage(err.data.ytMessage)
                  : decodeYPErrorMessage(errorMessage),
               ytError: decodeYPErrorMessages(err?.data?.ytError),
               autoDismiss: 0,
            });
            this.setState({
               editProjectIsSubmitting: false,
               isEditMode: false,
            });
         });
   }

   onCancelProjectEdit() {
      this.setState({ isEditMode: false });
   }

   onLoadMoreClick() {
      const { searchStage } = this.state;
      const { token } = this.getStagesToken() || {};
      this.setState({ stagesLoading: true });
      this.fetchStages({ token, substring: searchStage }).then(() => {
         this.setState({ stagesLoading: false });
      });
   }

   getTags() {
      const { projects } = this.props;
      const project = projects[this.projectId];

      return project?.labels?.tags ?? [];
   }

   getStagesToken() {
      const { continuationTokens } = this.props;
      const { searchStage } = this.state;
      const tokenProjectData = continuationTokens.stages[`${this.projectId}:${searchStage}`];
      if (!tokenProjectData) {
         return null;
      }
      const tokenData = tokenProjectData['/meta/id'];
      if (!tokenData) {
         return null;
      }
      return tokenData;
   }

   fetchProject() {
      return this.props.getProjects({
         ids: [this.projectId],
         paths: ['/labels', '/meta', '/spec', '/status'],
         fetchTimestamps: true,
      });
   }

   fetchAcl() {
      return this.props.getProjectAcl({
         params: {
            projectId: this.projectId,
         },
      });
   }

   fetchStages({ token, substring, reset = false } = {}) {
      return this.props.fetchStages({
         meta: { reset },
         limit: 50,
         project: this.projectId,
         substring,
         paths: s => [s.meta.id, s.status, s.labels, s.spec],
         continuationToken: token,
      });
   }

   openConfirmDialog(config) {
      this.setState({ confirmDialog: config });
   }

   closeConfirmDialog() {
      this.setState({ confirmDialog: null });
   }

   deleteProject() {
      const { notifications } = this.props;

      firstValueFrom(ypApi.deleteProject(this.projectId))
         .then(() => {
            projectsSlice.actions.deleteProject(this.projectId);
            this.props.history.push(urlBuilder.projects());
         })
         .catch(err => {
            console.error(err);
            let errorMessage = '';

            if (err?.data) {
               errorMessage = err.data.message || '';
            }

            notifications.add({
               level: 'error',
               title: `Delete project error`,
               message: err?.data?.ytMessage
                  ? decodeYPErrorMessage(err.data.ytMessage)
                  : decodeYPErrorMessage(errorMessage),
               ytError: decodeYPErrorMessages(err?.data?.ytError),
               autoDismiss: 0,
            });
         });
   }

   fetchData({ init = false } = {}) {
      this.fetchDataTimeout = window.setTimeout(() => this.fetchData(), 20000);

      // Issue requests concurrently, as they don't depend on each other
      Promise.all([this.fetchProject(), this.fetchStages({ reset: init }), this.fetchAcl()])
         .then(() => {
            const { projects } = this.props;
            const { spec } = projects[this.projectId];
            this.setState({
               isLoading: false,
               boxTypes: [...(spec.user_specific_box_types || [])],
            });
         })
         .catch(e => {
            console.error(e);
            this.setState({
               isLoading: false,
            });
         });
   }

   renderBreadcrumbs() {
      const {
         match: {
            params: { projectId },
         },
      } = this.props;
      return (
         <Breadcrumbs
            links={[
               { name: 'All', url: urlBuilder.projects() },
               { name: projectId, url: urlBuilder.project(projectId) },
            ]}
         />
      );
   }

   renderAbcService({ serviceId }) {
      const { abcServices } = this.props;
      if (!(serviceId in abcServices)) {
         if (!(serviceId in this.fetchedAbc)) {
            this.fetchedAbc[serviceId] = true;
            this.props.fetchAbcService({ params: { serviceId } }).catch(e => console.error(e));
         }

         return serviceId;
      }

      const service = abcServices[serviceId];

      return <ExternalLink href={EXTERNAL_LINKS.abcService(service.slug)}>{service.name}</ExternalLink>;
   }

   renderMonitoringProject(service) {
      return service ? <ExternalLink href={EXTERNAL_LINKS.monitoringService(service)}>{service}</ExternalLink> : '—';
   }

   renderGroup({ groupId }) {
      return <GroupLink groupId={groupId} />;
   }

   renderUser({ id }) {
      return <UserLink id={id} />;
   }

   renderManageRoleLink({ stageId, roleGroup, boxType }) {
      return (
         <ManageRoleLink
            {...{
               projectId: this.projectId,
               stageId,
               roleGroup,
               boxType,
            }}
         />
      );
   }

   renderRoleAccount({ account }) {
      return (
         <span key={account}>
            {account.match(/^group:\d+$/)
               ? this.renderGroup({ groupId: account.split(':')[1] })
               : this.renderUser({ id: account })}
         </span>
      );
   }

   renderRole({ role, accounts, inheritedAccounts, stageId, boxType }) {
      return (
         <div key={role} className={b('role-block')}>
            <div>
               <span className={b('role-title')}>{rolesData.roles[role]?.title ?? role}</span>
               <HelpPopover
                  openOnHover={false}
                  content={
                     <span>
                        {rolesData.roles[role]?.description ?? ''}{' '}
                        <ExternalLink href={EXTERNAL_LINKS.deployDocs.deployRoleModel}>docs</ExternalLink>
                     </span>
                  }
               />
            </div>
            <RolesList
               projectId={this.projectId}
               stageId={stageId}
               role={role}
               accounts={accounts}
               inheritedAccounts={inheritedAccounts}
               boxType={boxType}
            />
         </div>
      );
   }

   renderStageAcl() {
      const { projectsAcl, projects } = this.props;
      const { currentStageId, sideBarVisible } = this.state;
      if (!sideBarVisible) {
         return null;
      }
      const project = projects[this.projectId];
      const { spec } = project || {};
      if (currentStageId === '') {
         return null;
      }
      const inheritedAcl = ((projectsAcl[this.projectId] || {}).project_acl || {})[this.projectId] || {};
      const stageAcl = ((projectsAcl[this.projectId] || {}).stage_acl || {})[currentStageId] || {};
      const boxTypeAcl = ((projectsAcl[this.projectId] || {}).box_type_acl || {})[currentStageId] || {};
      const existAdditional =
         stageAclGroups.reduce((length, role) => length + (stageAcl[role] || []).length, 0) +
            (spec.user_specific_box_types || []).reduce(
               (length, type) =>
                  length + boxTypeAclGroups.reduce((l, role) => l + ((boxTypeAcl[type] || {})[role] || []).length, 0),
               0,
            ) >
         0;
      return (
         <SideBar
            collapse={() => this.setState({ sideBarVisible: false, currentStageId: '' })}
            className={b('sidebar')}
         >
            <div style={{ padding: '16px' }}>
               <section>
                  <h2>
                     <RouteLink to={urlBuilder.stage(currentStageId)} theme={'normal'}>
                        {currentStageId}
                     </RouteLink>
                  </h2>
                  {existAdditional && (
                     <div>
                        <div className={b('additional-caption')}>
                           <div className={b('additional-icon')}>
                              <i className={'fas fa-exclamation-circle'} />
                           </div>
                           <div>There are additional roles out of inherited group</div>
                        </div>
                     </div>
                  )}
                  <div className={b('acl-title')}>
                     <i className={'fas fa-rocket'} />
                     <h3>Stage</h3>
                  </div>
                  {stageAclGroups.map(role => (
                     <div key={role} className={b('acl-list')}>
                        {this.renderRole({
                           role,
                           accounts: stageAcl[role] || [],
                           stageId: currentStageId,
                           inheritedAccounts: inheritedAcl[role] || [],
                        })}
                     </div>
                  ))}
                  {/* https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9 */}
                  {spec.user_specific_box_types &&
                     false &&
                     spec.user_specific_box_types.map(type => (
                        <Fragment key={type}>
                           <div className={b('acl-title')}>
                              <i className={'fas fa-box-open'} />
                              <h3>{type}</h3>
                           </div>
                           {boxTypeAclGroups.map(role => (
                              <div key={role} className={b('acl-list')}>
                                 {this.renderRole({
                                    role,
                                    accounts: (boxTypeAcl[type] || {})[role] || [],
                                    stageId: currentStageId,
                                    boxType: type,
                                    inheritedAccounts: [
                                       ...new Set([...(inheritedAcl[role] || []), ...(stageAcl[role] || [])]),
                                    ],
                                 })}
                              </div>
                           ))}
                        </Fragment>
                     ))}
               </section>
            </div>
         </SideBar>
      );
   }

   renderStageCard({ stageId, additional, boxTypes, rolesCount }) {
      const { stages } = this.props;
      const { filterBoxType, searchStage, currentStageId, sideBarVisible } = this.state;
      return (
         <StageRolesCard
            key={stageId}
            stageId={stageId}
            stage={stages[stageId]}
            additional={additional}
            rolesCount={rolesCount}
            boxTypes={boxTypes}
            onRolesClick={() => {
               this.setState(state => ({
                  currentStageId: stageId,
                  sideBarVisible: state.currentStageId !== stageId || !state.sideBarVisible,
               }));
            }}
            filterBoxType={filterBoxType}
            searchStage={searchStage}
            currentStageId={currentStageId === stageId && !sideBarVisible ? '' : currentStageId}
         />
      );
   }

   renderStages() {
      const { stageNamesByProject, projectsAcl, projects } = this.props;
      const { stagesLoading } = this.state;

      const { terminator = false } = this.getStagesToken() || {};
      const stages = Object.keys(stageNamesByProject[this.projectId] || {});
      const project = projects[this.projectId];
      const { spec } = project || {};

      if (!(this.projectId in projectsAcl) || !spec) {
         return null;
      }

      const stageAcl = projectsAcl[this.projectId].stage_acl;
      const boxTypeAcl = projectsAcl[this.projectId].box_type_acl;
      return (
         <div>
            <div className={b('filters')}>
               <div className={b('filter', { type: 'name' })} data-test={'filter-projects-by-name'}>
                  <ColoredInput
                     text={this.state.searchStage || ''}
                     rules={{
                        object: [/^[a-zA-Z0-9-_]+/, b('highlight-object', { type: 'stage' })],
                     }}
                     placeholder={'Search by <stage-name>'}
                     onChange={value => this.setState({ searchStage: value })}
                     hasClear
                  />
               </div>
               <div>
                  {/* https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9 */}
                  {spec.user_specific_box_types && false && (
                     <div style={{ display: 'flex', alignItems: 'center' }}>
                        <span style={{ marginRight: '8px' }}>Box types</span>
                        <Select
                           val={this.state.filterBoxType}
                           items={[
                              { val: '', text: 'all' },
                              ...spec.user_specific_box_types.map(type => ({
                                 val: type,
                                 key: type,
                                 text: type,
                              })),
                           ]}
                           onChange={value => this.setState({ filterBoxType: value[0] })}
                        />
                     </div>
                  )}
               </div>
            </div>
            <div className={b('stage-list')}>
               {stages
                  .filter(stageId => {
                     const includesName = stageId.includes(this.state.searchStage || '');
                     const includesBoxType =
                        !this.state.filterBoxType ||
                        this.state.filterBoxType === 'all' ||
                        Object.keys(boxTypeAcl[stageId] || {}).includes(this.state.filterBoxType);
                     return includesName && includesBoxType;
                  })
                  .sort()
                  .map(stageId => {
                     const additional = stageId in stageAcl || stageId in boxTypeAcl;
                     let count = 0;
                     if (additional) {
                        // TODO сложно для чтения, переделать
                        count =
                           Object.values(stageAcl[stageId] || {}).reduce((s, e) => s + e.length, 0) +
                           Object.values(boxTypeAcl[stageId] || {}).reduce(
                              // eslint-disable-next-line no-shadow
                              (s, e) => s + Object.values(e).reduce((s, e) => s + e.length, 0),
                              0,
                           );
                     }
                     const boxTypes = Object.keys(boxTypeAcl[stageId] || {});
                     return this.renderStageCard({ stageId, additional, rolesCount: count, boxTypes });
                  })}
            </div>
            {!terminator && (
               <div>
                  {stagesLoading ? (
                     <Spin progress size={'xxs'} />
                  ) : (
                     <Button theme={'link'} onClick={this.onLoadMoreClick.bind(this)}>
                        Load more
                     </Button>
                  )}
               </div>
            )}
         </div>
      );
   }

   renderStagesAcl() {
      const { stageNamesByProject } = this.props;
      const stages = Object.keys(stageNamesByProject[this.projectId] || {});
      return (
         <>
            <div style={{ marginTop: '24px' }}>
               <h2>Stage access</h2>
               <Note
                  text={
                     'It’s possible to expand permissions for particular stage elements. Click Roles for additional settings.'
                  }
               />
            </div>
            {stages.length ? (
               <div>
                  {this.renderStages()}
                  {this.renderStageAcl()}
               </div>
            ) : (
               <EmptyContainer
                  type={EmptyContainerType.EmptyState}
                  title={'No stages created yet'}
                  description={'You can try to create it.'}
                  action={{
                     text: 'Create new Stage',
                     link: urlBuilder.newStage(this.projectId),
                  }}
               />
            )}
         </>
      );
   }

   renderProjectAcl() {
      const { projectsAcl } = this.props;

      const acl = projectsAcl?.[this.projectId]?.project_acl?.[this.projectId];

      return (
         <div style={{ maxWidth: '700px' }}>
            <h2>General access</h2>

            {!acl ? (
               <div className={b('no-available-acl')} data-e2e={'no-available-acl'}>
                  No available acl. IDM will be updated in about 5 minutes. <Spin progress size={'xxs'} />
               </div>
            ) : (
               <>
                  <Note
                     text={
                        <>
                           Following permissions are relevant for all stages within this project (excluding system box),
                           use quick links below to change ACL through IDM. Read more about roles and permissions in
                           Y.Deploy on{' '}
                           <ExternalLink href={EXTERNAL_LINKS.deployDocs.deployRoleModel}>
                              documentation page
                           </ExternalLink>
                        </>
                     }
                  />
                  {projectAclGroups.map(role => this.renderRole({ role, accounts: acl[role] || [] }))}
                  <ButtonLink
                     to={EXTERNAL_LINKS.idmRequestDeployRole({
                        system: getConfig().idmSystemName,
                        role: rolesData.roles.viewer.idmId,
                        projectId: this.projectId,
                        groups: [202613],
                        expireDays: 7,
                        comment: 'Shortcut request from GUI',
                     })}
                     view={'action'}
                     external={true}
                  >
                     Request support VIEWER role for 7 days
                  </ButtonLink>
               </>
            )}
         </div>
      );
   }

   renderHeader() {
      const { projects } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      if (!spec) {
         return null;
      }
      return (
         <div className={b('header-container')}>
            <div>{this.renderBreadcrumbs()}</div>
            <div className={b('update-time')}>
               Updated on {dateFormat(new Date(spec[Symbol.for('timestamp')]), 'mmm d, yyyy, H:MM')}
            </div>
            <div className={b('header')}>
               <div>
                  <h1 className={b('title')}>{this.projectId}</h1>
               </div>
               <div
                  ref={elem => {
                     this.buttonsContainerElement = elem;
                  }}
                  className={b('actions-container')}
               >
                  <div className={b('actions')}>
                     {['edit', 'delete'].map(name => {
                        const action = this.actions[name];
                        const active = action.isActive ? action.isActive() : true;

                        return (
                           <div key={name} className={b('action-container')} data-test={`project-actions__${name}`}>
                              <button
                                 className={b('action-button', { active })}
                                 onClick={action.action}
                                 type={'button'}
                              >
                                 <i className={`far fa-${action.icon}`} />
                              </button>
                           </div>
                        );
                     })}
                  </div>

                  <div data-test={'project-actions__create-stage'}>
                     <ButtonLink to={urlBuilder.newStage(this.projectId)} view={'action'}>
                        Create new Stage
                     </ButtonLink>
                  </div>
               </div>
            </div>
         </div>
      );
   }

   renderEditForm() {
      const { projects, abcServices } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      const abcService = abcServices[spec.account_id.split(':')[2]];
      return (
         <div className={b('edit-form')}>
            <ProjectEditForm
               submitText={'Save'}
               onSubmit={values => {
                  this.setState(
                     {
                        editProjectIsSubmitting: true,
                     },
                     () => this.onSaveProjectEdit(values),
                  );
               }}
               handleCancel={this.onCancelProjectEdit.bind(this)}
               initialValues={{
                  box_types: (spec.user_specific_box_types || []).join(','),
                  new_box_types: '',
                  delete_box_types: '',
                  new_box_type: '',
                  abc: (id =>
                     id && id !== 'tmp'
                        ? {
                             name: abcService ? abcService.name : id.split(':')[2],
                             _id: id.split(':')[2],
                          }
                        : null)(spec.account_id),
                  temporary_account: spec.account_id === 'tmp',
                  tags: this.getTags(),
                  monitoring_project: spec.monitoring_project,
               }}
               isSubmitting={this.state.editProjectIsSubmitting || false}
            />
         </div>
      );
   }

   renderAbcView() {
      const { projects } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      return (
         <div>
            <div className={b('abc-line')}>
               <div>ABC Service:</div>
               <div>
                  {spec.account_id === 'tmp'
                     ? 'tmp'
                     : this.renderAbcService({ serviceId: spec.account_id.split(':')[2] })}
               </div>
               {spec.account_id !== 'tmp' && (
                  <Note
                     text={
                        <>
                           Physical resources (RAM, CPU, etc.) for all stages in this project are allocated in quotas
                           that are granted to the respective ABC service. Read more about quotas in{' '}
                           <ExternalLink href={EXTERNAL_LINKS.docs.rtcQuotas}>documentation</ExternalLink>
                        </>
                     }
                  />
               )}
            </div>
            {spec.account_id === 'tmp' && (
               <div className={b('abc-warning')}>
                  <WarningPlate>
                     Project is using temporary quota. Y.Deploy won&#39;t guarantee availability of stages in this
                     project, they can be deleted at any time. You have to use this option for testing only. We are
                     going to send notifications about using temporary quota to every user in abc-groups, that are
                     listed in an ACL.{' '}
                     <ExternalLink href={EXTERNAL_LINKS.docs.rtcQuotas}>YP Quota documentation</ExternalLink>
                     <br />
                     <br />
                     More importantly we are planning to abandon temporary quota in favour of Free Tier Quota.
                     <br />
                     <ExternalLink href={'https://clubs.at.yandex-team.ru/infra-cloud/1269'}>
                        Announcement
                     </ExternalLink>{' '}
                     <ExternalLink href={EXTERNAL_LINKS.wiki.ypFreeTier}>Free Tier documentation</ExternalLink>
                  </WarningPlate>
               </div>
            )}
         </div>
      );
   }

   renderMonitoringInfo() {
      const { projects } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      return (
         <div>
            <div className={b('monitoring-line')}>
               Monitoring project: <span>{this.renderMonitoringProject(spec.monitoring_project)}</span>
            </div>
            {spec.account_id !== 'tmp' && !spec.monitoring_project && (
               <div className={b('monitoring-project-warning')}>
                  <WarningPlate>Monitoring project is not configured</WarningPlate>
               </div>
            )}
         </div>
      );
   }

   renderTags() {
      const tags = new Set(this.getTags());

      return (
         <div className={b('tags-line')}>
            Tags
            <span className={b('tags-hint')}>
               <Hint text={<TagsHint />} />
            </span>
            :{' '}
            {tags.size > 0 ? (
               <TagList editable={false} items={setToListOptions(tags)} className={b('tags-line-list')} />
            ) : (
               <Note text={'no tags'} />
            )}
         </div>
      );
   }

   renderProjectSpec() {
      const { projects } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      const { isEditMode } = this.state;
      if (isEditMode) {
         return this.renderEditForm();
      }
      return (
         <>
            {this.renderAbcView()}
            {this.renderMonitoringInfo()}
            {this.renderTags()}
            {/* TODO https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9 */}
            {false && (
               <div>
                  Box types: {(spec.user_specific_box_types || []).join(', ')}&nbsp;
                  <Tooltip
                     text={this.state.boxTypesEdit ? this.renderBoxTypesEdit() : null}
                     visible={this.state.boxTypesEdit}
                     clickable={true}
                     modal={true}
                  >
                     <div
                        className={b('action-button', { active: this.state.boxTypesEdit })}
                        role={'button'}
                        tabIndex={0}
                        aria-label={'Edit'}
                        onClick={() => this.setState({ boxTypesEdit: true })}
                     >
                        <i className={`far fa-${this.actions.edit.icon}`} />
                     </div>
                  </Tooltip>
               </div>
            )}
         </>
      );
   }

   renderProjectInfo() {
      const { projects } = this.props;
      const project = projects[this.projectId];
      const { spec } = project || {};
      if (!project || !spec) {
         return null;
      }
      return (
         <div>
            {this.renderProjectSpec()}
            {this.renderProjectAcl()}
         </div>
      );
   }

   renderConfirmDialog() {
      const { confirmDialog } = this.state;

      if (!confirmDialog) {
         return null;
      }

      return (
         <ConfirmDialog
            visible={true}
            title={confirmDialog.title}
            message={confirmDialog.message}
            actionText={confirmDialog.actionText}
            actionTheme={confirmDialog.actionTheme}
            actionHandler={confirmDialog.actionHandler}
            cancelText={confirmDialog.cancelText}
            confirmation={confirmDialog.confirmation}
            onRequestClose={this.closeConfirmDialog.bind(this)}
         />
      );
   }

   render() {
      if (this.state.isLoading) {
         return (
            <div className={b('content-spinner')}>
               <Spin size={'l'} progress />
            </div>
         );
      }
      if (!this.props.projects[this.projectId]) {
         return (
            <EmptyContainer
               type={EmptyContainerType.NotFound}
               title={'Project was not found'}
               action={{ text: 'Back to Home page', link: urlBuilder.home() }}
            />
         );
      }
      return (
         <PageWrapper title={`Project: ${this.projectId}`}>
            <Page className={b()} dataTest={'page-project'}>
               {this.renderHeader()}
               {this.renderProjectInfo()}
               {this.renderStagesAcl()}
               {this.renderConfirmDialog()}
            </Page>
         </PageWrapper>
      );
   }
}

const mapStateToProps = createStructuredSelector({
   projects: selectProjects,
   abcServices: getAbcServices,
   clusters: getClusters,
   projectsAcl: selectProjectsAcl,
   stages: selectStages,
   stageNamesByProject: selectStageNamesByProject,
   fetchedGroups: selectFetchedGroups,
   groups: selectGroups,
   continuationTokens: selectContinuationTokens,
});

const mapDispatchToProps = {
   fetchAbcService,
   getClusters,
   getProjectAcl,
   getProjects,
   fetchStages: ({ meta, ...params }) => fetchStages.withMeta(meta)({ ...params }),
   fetchGroups,
   deleteProject,
};

export default compose(withRouter, withNotifications, connect(mapStateToProps, mapDispatchToProps))(Project);
