import React, { Component } from 'react';
import { connect } from 'react-redux';
import { compose } from 'recompose';
import { createStructuredSelector } from 'reselect';
import {
   fetchStages,
   getStages,
   selectAvailableObjects,
   selectContinuationTokens,
   selectProject,
   selectStageNamesByProject,
   selectStages,
} from '../../../redux';
import { getConfig } from '../../../services/Config';

import ProjectCard from '../../components/ProjectCard/ProjectCard';

class ProjectCardContainer extends Component {
   state = {
      isLoading: false,
      checkStages: false,
      filters: {
         stage: '',
      },
   };

   searchCache = {
      stage: {},
   };

   myStageTokens = {};

   myStageEnd = {};

   onSearch({ name }) {
      this.setState(
         {
            filters: {
               stage: name,
            },
         },
         () => {
            this.fetchFilteredStagesWithCache({ name: this.getFilterStageName() });
         },
      );
   }

   onLoadButtonClick() {
      const name = this.getFilterStageName();
      this.fetchFilteredStages({ name });
   }

   onProjectHeaderClick() {
      const { toggleCard, opened } = this.props;

      if (opened) {
         toggleCard();

         return;
      }

      const name = this.getFilterStageName();

      let load = true;

      const setLoadFlags = () => {
         load = false;
         this.setState({ checkStages: true });
      };

      window.setTimeout(() => (load ? this.setState({ isLoading: true }) : null), 0);
      const checkStages = () => {
         const { stageNamesByProject, projectName } = this.props;
         if ((Object.keys(stageNamesByProject[projectName] || {}) || []).length > 0) {
            toggleCard();
         }
         this.setState({ isLoading: false });
      };
      this.fetchFilteredStagesWithCache({ name })
         .then(() => {
            setLoadFlags();
            window.setTimeout(() => checkStages(), 50);
         })
         .catch(e => {
            console.error(e);
            setLoadFlags();
            toggleCard();
            this.setState({ isLoading: false });
         });
   }

   getFilterStageName() {
      const {
         filters: { stage },
      } = this.props;
      const {
         filters: { stage: stageName },
      } = this.state;
      return stage || stageName || '';
   }

   fetchFilteredStages({ name }) {
      const {
         projectName,
         continuationTokens,
         filters: { my },
         objectIdsWithLogins,
         stages,
      } = this.props;
      const limit = 100; // столько карточек будет открываться за раз
      if (my === 'yes') {
         const stageIds = Object.keys(objectIdsWithLogins.stage || {});
         const myStageIds = stageIds.filter(
            stageId => ((objectIdsWithLogins.stage || {})[stageId] || {})[getConfig().user.login],
         );
         const myFilteredStageIds = myStageIds.filter(
            stageId => ((stages[stageId] || {}).meta || {}).project_id === projectName && stageId.includes(name), // before migration
         );
         const token = this.myStageTokens[name];
         const startIndex = token ? myFilteredStageIds.indexOf(token) : 0;
         const endIndex = startIndex + limit;
         const currentPart = myFilteredStageIds.slice(startIndex, endIndex);
         if (endIndex < myFilteredStageIds.length) {
            this.myStageTokens[name] = myFilteredStageIds[endIndex];
         } else {
            this.myStageEnd[name] = true;
         }
         return this.props.getStages({
            objectIds: currentPart,
            paths: ['/labels', '/meta', '/spec', '/status'],
            fetchTimestamps: true,
         });
      }
      return this.props.fetchStages({
         limit,
         project: projectName,
         substring: name,
         continuationToken: ((continuationTokens.stages[`${projectName}:${name}`] || {})['/spec'] || {}).token,
      });
   }

   fetchFilteredStagesWithCache({ name }) {
      const lastSearchTimeout = this.searchCache.stage[name];
      const isLongPeriod = () => new Date() - lastSearchTimeout > 10000; // 10 sec
      const searchAvailable = !lastSearchTimeout || isLongPeriod();
      if (searchAvailable) {
         this.searchCache.stage[name] = new Date();
         return this.fetchFilteredStages({ name });
      }
      return Promise.resolve();
   }

   existMoreStages() {
      const {
         filters: { my },
         projectName,
         continuationTokens,
      } = this.props;
      const name = this.getFilterStageName();
      if (my === 'yes') {
         return !this.myStageEnd[name];
      }
      return !((continuationTokens.stages[`${projectName}:${name}`] || {})['/spec'] || {}).terminator;
   }

   render() {
      const {
         opened,
         stages,
         projectName,
         filters: globalFilters,
         objectIdsWithLogins,
         endOpened,
         startOpened,
         stageNamesByProject,
         project,
      } = this.props;
      const { isLoading, checkStages, filters } = this.state;

      const tags = project?.labels?.tags ?? [];

      return (
         <ProjectCard
            projectName={projectName}
            opened={opened}
            isLoading={isLoading}
            checkStages={checkStages}
            startOpened={startOpened}
            endOpened={endOpened}
            globalFilters={globalFilters}
            filters={filters}
            stages={stages}
            tags={tags}
            existMoreStages={this.existMoreStages()}
            stageNamesByProject={stageNamesByProject}
            objectIdsWithLogins={objectIdsWithLogins}
            onProjectHeaderClick={this.onProjectHeaderClick.bind(this)}
            onSearch={this.onSearch.bind(this)}
            onLoadButtonClick={this.onLoadButtonClick.bind(this)}
         />
      );
   }
}

const mapStateToProps = createStructuredSelector({
   stages: selectStages,
   continuationTokens: selectContinuationTokens,
   objectIdsWithLogins: selectAvailableObjects,
   stageNamesByProject: selectStageNamesByProject,
   project: (state, { projectName }) => selectProject(state, projectName),
});

const mapDispatchToProps = {
   fetchStages,
   getStages,
};

export default compose(connect(mapStateToProps, mapDispatchToProps))(ProjectCardContainer);
