import { Button } from '@yandex-cloud/uikit';
import { ExternalLink } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import throttle from 'lodash/throttle';

import PropTypes from 'prop-types';
import React from 'react';
import { connect } from 'react-redux';
import { compose } from 'recompose';
import { Field, formValueSelector, reduxForm } from 'redux-form';
import { handlePreventDefault } from '../../../utils';

import { EXTERNAL_LINKS } from '../../../models';

import Form from '../../components/Form/Form';
import FormFieldCheckbox from '../../components/FormFieldCheckbox/FormFieldCheckbox';
import FormFieldSuggest from '../../components/FormFieldSuggest/FormFieldSuggest';
import FormFieldText from '../../components/FormFieldText/FormFieldText';
import FormLayout from '../../components/FormLayout/FormLayout';

import withNotifications from '../../components/hoc/withNotifications';

import Note from '../../components/Note/Note';

import { fetchQuotaGroups } from '../../store/reducers/awacs';

import './ProjectCreateForm.scss';

const b = block('project-create-form');

let formErrorTimeout = false;

class ProjectCreateForm extends React.Component {
   static propTypes = {
      notifications: PropTypes.object.isRequired,
      error: PropTypes.object,
      valid: PropTypes.bool.isRequired,
      submitting: PropTypes.bool.isRequired,
      handleSubmit: PropTypes.func.isRequired,
      handleCancel: PropTypes.func.isRequired,
      isSubmitting: PropTypes.bool.isRequired,
      submitText: PropTypes.string.isRequired,
   };

   state = {
      quotaGroupsSuggestData: [],
   };

   componentDidMount() {
      this.throttledFetchQuotaGroups = throttle(this.fetchQuotaGroups, 300);
   }

   onQuotaGroupsChange(element, item) {
      const query = item.name ? item.name : item;
      this.quotaGroupsQuery = query.trim();
      if (this.quotaGroupsQuery.length === 0) {
         this.setState({
            quotaGroupsSuggestData: [],
         });
         return;
      }
      this.throttledFetchQuotaGroups(this.quotaGroupsQuery);
   }

   fetchQuotaGroups(query) {
      fetchQuotaGroups({
         params: {
            query,
         },
      }).then(response => {
         if (this.quotaGroupsQuery === query) {
            this.setState({
               quotaGroupsSuggestData: response,
            });
         }
      });
   }

   showErrors() {
      const { notifications, valid } = this.props;
      Object.keys(notifications.notificationsData).forEach(notificationId => {
         notifications.remove(notificationId);
      });
      if (!valid) {
         if (!formErrorTimeout) {
            notifications.add({
               level: 'error',
               title: 'Can not submit',
               message: 'Check your form for errors before submitting.',
            });
         }
         formErrorTimeout = true;
         window.setTimeout(() => {
            formErrorTimeout = false;
         }, 3000);
      }
   }

   renderButtons() {
      const { handleCancel, handleSubmit, initialValue, isSubmitting, submitText } = this.props;

      return (
         <div className={b('buttons-container')}>
            <div className={b('buttons')}>
               <span data-test={'project-create-form__button_cancel'}>
                  <Button
                     onClick={event => {
                        event.preventDefault();
                        this.props.initialize(initialValue);
                        handleCancel();
                     }}
                     view={'clear'}
                  >
                     Cancel
                  </Button>
               </span>
               <span data-test={'project-create-form__button_submit'}>
                  <Button
                     onClick={event => {
                        event.preventDefault();
                        this.showErrors();
                        handleSubmit();
                     }}
                     view={'action'}
                     loading={isSubmitting}
                  >
                     {submitText}
                  </Button>
               </span>
            </div>
         </div>
      );
   }

   render() {
      const { handleSubmit, handleCancel, isSubmitting, fieldTemporaryAccount, submitText } = this.props;
      const { quotaGroupsSuggestData } = this.state;
      return (
         <Form
            submitText={submitText}
            onSubmit={handlePreventDefault}
            onSubmitClick={event => {
               event.preventDefault();
               this.showErrors();
               handleSubmit();
            }}
            isSubmitting={isSubmitting}
            className={b()}
            dataTest={'project-create-form'}
            autoComplete={'off'}
            onCancelClick={event => {
               event.preventDefault();
               handleCancel();
            }}
            customButtons
         >
            <Form.Section>
               <FormLayout className={b('content')}>
                  <FormLayout.Row title={'Project Name'} className={b('field_id')}>
                     <Field name={'project_name'} component={FormFieldText} dataTest={'form-field--project-name'} />
                     <Note text={'Project Name can not be changed after creation'} />
                  </FormLayout.Row>
                  <FormLayout.Row
                     title={'ABC Service'}
                     note={'abcService'}
                     className={`${b('suggest-field')} ${b('field_abc')}`}
                  >
                     <Field
                        name={'abc'}
                        component={FormFieldSuggest}
                        onChange={this.onQuotaGroupsChange.bind(this)}
                        data={quotaGroupsSuggestData}
                        filter={false}
                        filterValueBy={'name'}
                        filterBy={'name'}
                        isCommaSeparated={false}
                        placeholder={'Type to search...'}
                        disabled={fieldTemporaryAccount}
                        dataTest={'form-field--abc-service'}
                        noCustomValue={true}
                     />
                  </FormLayout.Row>
                  <FormLayout.Row className={`${b('field_account-tmp')}`}>
                     <Field
                        name={'temporary_account'}
                        text={'Use temporary account'}
                        component={FormFieldCheckbox}
                        dataTest={'form-field--use-temporary-account'}
                     />
                     {fieldTemporaryAccount && (
                        <Note
                           text={
                              <>
                                 When using temporary quota Y.Deploy won&#39;t guarantee your stage&#39;s availability
                                 and liveness, it can be deleted at any time. More importantly we are planning to
                                 abandon temporary quota in favour of Free Tier Quota.
                                 <br />
                                 <ExternalLink href={'https://clubs.at.yandex-team.ru/infra-cloud/1269'}>
                                    announcement
                                 </ExternalLink>
                                 <ExternalLink href={EXTERNAL_LINKS.wiki.ypFreeTier}>documentation</ExternalLink>
                              </>
                           }
                        />
                     )}
                     <Note
                        text={
                           <>
                              Physical resources (RAM, CPU, etc.) for all stages in this project will be allocated in
                              quotas that are granted to the respective ABC service. Read more about Quotas in{' '}
                              <ExternalLink className={b('note-link')} href={EXTERNAL_LINKS.docs.rtcQuotas}>
                                 documentation.
                              </ExternalLink>
                           </>
                        }
                     />
                  </FormLayout.Row>
               </FormLayout>
            </Form.Section>
            {this.renderButtons()}
         </Form>
      );
   }
}

/* eslint-disable camelcase, complexity */
const validate = (values, props) => {
   const errors = {};
   if (!values.project_name) {
      errors.project_name = 'Project name is required';
   } else if (!values.project_name.match(/^[A-Za-z0-9-_]+$/)) {
      errors.project_name = 'This project name must match regexp /[A-Za-z0-9-_]+/';
   } else if (props.projects[values.project_name]) {
      errors.project_name = 'This project name is already in use';
   }
   if (!values.temporary_account) {
      if (!values.abc) {
         errors.abc = 'ABC Service is required';
         // eslint-disable-next-line no-underscore-dangle
      } else if (!values.abc._id) {
         errors.abc = 'Select ABC Service from the drop-down list';
      }
   }
   return errors;
};

const selector = formValueSelector('create-project');

export default compose(
   withNotifications,
   reduxForm({
      form: 'create-project',
      validate,
      touchOnBlur: false,
      touchOnChange: false,
   }),
   connect(state => ({
      fieldTemporaryAccount: selector(state, 'temporary_account'),
   })),
)(ProjectCreateForm);
