import { Button } from '@yandex-cloud/uikit';
import block from 'bem-cn-lite';
import throttle from 'lodash/throttle';
import PropTypes from 'prop-types';
import React, { useCallback, useMemo } from 'react';
import { firstValueFrom } from 'rxjs';

import { connect } from 'react-redux';
import { compose } from 'recompose';
import { Field, formValueSelector, reduxForm } from 'redux-form';
import { TagInputEnhanced } from '../../../components/forms/inputs/TagInput/TagInput';
import { projectManagerApi } from '../../../services';
import { handlePreventDefault } from '../../../utils';
import Form from '../../components/Form/Form';
import FormFieldCheckbox from '../../components/FormFieldCheckbox/FormFieldCheckbox';
import FormFieldSuggest from '../../components/FormFieldSuggest/FormFieldSuggest';
import FormFieldText from '../../components/FormFieldText/FormFieldText';
import FormLayout from '../../components/FormLayout/FormLayout';

import withNotifications from '../../components/hoc/withNotifications';

import { fetchQuotaGroups } from '../../store/reducers/awacs';

import './ProjectEditForm.scss';

const b = block('project-edit-form');

let formErrorTimeout = false;

class ProjectEditForm extends React.Component {
   static propTypes = {
      notifications: PropTypes.object.isRequired,
      error: PropTypes.object,
      valid: PropTypes.bool.isRequired,
      change: PropTypes.func.isRequired,
      submitting: PropTypes.bool.isRequired,
      handleSubmit: PropTypes.func.isRequired,
      handleCancel: PropTypes.func.isRequired,
      isSubmitting: PropTypes.bool.isRequired,
      submitText: PropTypes.string.isRequired,
      fieldBoxTypes: PropTypes.string.isRequired,
      fieldNewBoxTypes: PropTypes.string.isRequired,
      fieldDeleteBoxTypes: PropTypes.string.isRequired,
      fieldNewBoxType: PropTypes.string.isRequired,
   };

   cardSettings = {
      new: {
         field: 'new_box_types',
         prop: 'fieldNewBoxTypes',
         delete: type => {
            const field = this.props.fieldNewBoxTypes;
            console.log({ field });
            const list = field ? field.split(',') : [];
            const i = list.findIndex(e => e === type);
            list.splice(i, 1);
            this.props.change('new_box_types', list.join(','));
         },
      },
      current: {
         field: 'box_types',
         prop: 'fieldBoxTypes',
         delete: type => {
            const field = this.props.fieldDeleteBoxTypes;
            const list = field ? field.split(',') : [];
            this.props.change('delete_box_types', list.concat(type).join(','));
         },
         revert: type => {
            const field = this.props.fieldDeleteBoxTypes;
            const list = field ? field.split(',') : [];
            const i = list.findIndex(e => e === type);
            list.splice(i, 1);
            this.props.change('delete_box_types', list.join(','));
         },
      },
   };

   state = {
      quotaGroupsSuggestData: [],
      monitoringProjectSuggestData: [],
   };

   componentDidMount() {
      this.throttledFetchQuotaGroups = throttle(this.fetchQuotaGroups, 300);
      this.throttledFetchMonitorings = throttle(this.fetchMonitoringProjects, 300);
   }

   onNewBoxTypeKeyUp(event) {
      const { fieldNewBoxTypes, fieldNewBoxType } = this.props;
      if (event.keyCode === 13) {
         if (fieldNewBoxType === '') {
            return;
         }
         if ((fieldNewBoxTypes || '').split(',').includes(fieldNewBoxType)) {
            return;
         }
         this.props.change('new_box_types', (fieldNewBoxTypes ? `${fieldNewBoxTypes},` : '') + fieldNewBoxType);
         this.props.change('new_box_type', '');
      }
   }

   onQuotaGroupsChange(element, item) {
      const query = item.name ? item.name : item;
      this.quotaGroupsQuery = query.trim();
      if (this.quotaGroupsQuery.length === 0) {
         this.setState({
            quotaGroupsSuggestData: [],
            monitoringProjectSuggestData: [],
         });
         return;
      }
      this.throttledFetchQuotaGroups(this.quotaGroupsQuery);
   }

   onMonitoringChange(element, item) {
      const query = item.name ? item.name : item;
      this.monitoringQuery = query.trim();
      if (this.monitoringQuery.length === 0) {
         this.setState({
            monitoringProjectSuggestData: [],
         });
         return;
      }

      this.throttledFetchMonitorings(this.monitoringQuery);
   }

   fetchQuotaGroups(query) {
      fetchQuotaGroups({
         params: {
            query,
         },
      }).then(response => {
         if (this.quotaGroupsQuery === query) {
            this.setState({
               quotaGroupsSuggestData: response,
            });
         }
      });
   }

   fetchMonitoringProjects(query) {
      firstValueFrom(projectManagerApi.getProjects(query))
         .catch(e => {
            console.log(e);
            return { projects: [] };
         })
         .then(response => {
            if (this.monitoringQuery === query) {
               this.setState({
                  monitoringProjectSuggestData: response.projects,
               });
            }
         });
   }

   showErrors() {
      const { notifications, valid } = this.props;
      Object.keys(notifications.notificationsData).forEach(notificationId => {
         notifications.remove(notificationId);
      });
      if (!valid) {
         if (!formErrorTimeout) {
            notifications.add({
               level: 'error',
               title: 'Can not submit',
               message: 'Check your form for errors before submitting.',
            });
         }
         formErrorTimeout = true;
         window.setTimeout(() => {
            formErrorTimeout = false;
         }, 3000);
      }
   }

   renderButtons() {
      const { handleSubmit, isSubmitting, submitText, handleCancel, fieldNewBoxType, fieldNewBoxTypes } = this.props;

      return (
         <div className={b('buttons-container')}>
            <div className={b('buttons')}>
               <Button
                  onClick={event => {
                     event.preventDefault();
                     handleCancel();
                  }}
                  view={'clear'}
                  qa={'ProjectEditForm:CancelButton'}
               >
                  Cancel
               </Button>
               <Button
                  onClick={event => {
                     event.preventDefault();
                     this.showErrors();
                     if (fieldNewBoxType) {
                        this.props.change(
                           'new_box_types',
                           (fieldNewBoxTypes ? `${fieldNewBoxTypes},` : '') + fieldNewBoxType,
                        );
                        this.props.change('new_box_type', '');
                        setTimeout(() => handleSubmit(), 300);
                     } else {
                        handleSubmit();
                     }
                  }}
                  view={'action'}
                  loading={isSubmitting}
                  qa={'ProjectEditForm:SubmitButton'}
               >
                  {submitText}
               </Button>
            </div>
         </div>
      );
   }

   renderBoxTypeCard({ type, category }) {
      const { fieldDeleteBoxTypes } = this.props;
      const isDelete = category === 'current' && fieldDeleteBoxTypes.split(',').includes(type);
      const settings = this.cardSettings[category];

      return (
         <span key={type} className={b('label', { [category]: true, delete: isDelete })}>
            <span>{type}</span>
            <span
               className={b('delete-icon')}
               onClick={() => settings[isDelete ? 'revert' : 'delete'].bind(this)(type)}
               role={'button'}
               aria-label={isDelete ? 'Revert' : 'Delete'}
               tabIndex={0}
            >
               <i className={isDelete ? 'far fa-undo-alt' : 'fal fa-times'} />
            </span>
         </span>
      );
   }

   render() {
      const {
         handleSubmit,
         handleCancel,
         isSubmitting,
         submitText,
         fieldBoxTypes,
         fieldNewBoxTypes,
         fieldTemporaryAccount,
         fieldAbc,
         initialValues,
      } = this.props;
      const { quotaGroupsSuggestData, monitoringProjectSuggestData } = this.state;

      return (
         <Form
            submitText={submitText}
            onSubmit={handlePreventDefault}
            onSubmitClick={event => {
               event.preventDefault();
               this.showErrors();
               handleSubmit();
            }}
            isSubmitting={isSubmitting}
            className={b()}
            dataTest={'project-edit-form'}
            autoComplete={'off'}
            onCancelClick={event => {
               event.preventDefault();
               handleCancel();
            }}
            customButtons
         >
            <Form.Section>
               <FormLayout>
                  <FormLayout.Row
                     title={'ABC Service'}
                     note={'abcService'}
                     className={`${b('suggest-field')} ${b('field_abc')}`}
                  >
                     <Field
                        name={'abc'}
                        component={FormFieldSuggest}
                        onChange={this.onQuotaGroupsChange.bind(this)}
                        data={quotaGroupsSuggestData}
                        filter={false}
                        filterValueBy={'name'}
                        filterBy={'name'}
                        isCommaSeparated={false}
                        placeholder={'Type to search...'}
                        disabled={fieldTemporaryAccount}
                        dataTest={'form-field--abc-service'}
                     />
                  </FormLayout.Row>
                  <FormLayout.Row className={`${b('field_account-tmp')}`}>
                     <Field
                        name={'temporary_account'}
                        text={'Use temporary account'}
                        component={FormFieldCheckbox}
                        dataTest={'form-field--use-temporary-account'}
                     />
                  </FormLayout.Row>

                  <FormLayout.Row title={'Monitoring project'} note={'monitoringProject'}>
                     <Field
                        name={'monitoring_project'}
                        component={FormFieldSuggest}
                        onChange={this.onMonitoringChange.bind(this)}
                        data={monitoringProjectSuggestData}
                        filter={false}
                        filterValueBy={'name'}
                        filterBy={'name'}
                        renderItem={(item, title) => (
                           <div className={b('monitoring-project__item')}>
                              <div>{item[title]}</div>
                              <div className={b('monitoring-project__item__meta')}>project ID: {item.projectId}</div>
                           </div>
                        )}
                        isCommaSeparated={false}
                        disabled={fieldTemporaryAccount}
                        dataTest={'form-field--monitoring-project'}
                     />
                  </FormLayout.Row>

                  <FormLayout.Row title={'Tags'} note={'tags'}>
                     <Field name={'tags'} component={FormFieldTags} />
                  </FormLayout.Row>

                  {(fieldTemporaryAccount
                     ? !initialValues.temporary_account
                     : // eslint-disable-next-line no-underscore-dangle
                       (fieldAbc || {})._id !== (initialValues.abc || {})._id) && (
                     <div className={b('warning', { 'change-abc': true })}>
                        Changing ABC Service means changing quota provider. You should re-deploy all stages in this
                        project manually to move them into the quota of the new ABC Service.
                     </div>
                  )}

                  {/* TODO https://st.yandex-team.ru/DEPLOY-2048#5e74f20b0b67f4308ae043c9 */}
                  {false && (
                     <>
                        <div style={{ display: 'none' }}>
                           <Field name={'box_types'} component={'input'} />
                           <Field name={'new_box_types'} component={'input'} />
                           <Field name={'delete_box_types'} component={'input'} />
                        </div>
                        <div>
                           It’s impossible to change box type after creation:
                           <div
                              style={{ margin: '8px 4px' }}
                              onKeyUp={this.onNewBoxTypeKeyUp.bind(this)}
                              role={'textbox'}
                              tabIndex={0}
                           >
                              <Field
                                 name={'new_box_type'}
                                 component={FormFieldText}
                                 placeholder={'Enter new box type'}
                              />
                           </div>
                           <div style={{ marginTop: '8px' }}>
                              {fieldBoxTypes &&
                                 (fieldBoxTypes.split(',') || []).map(type =>
                                    this.renderBoxTypeCard({
                                       type,
                                       category: 'current',
                                    }),
                                 )}
                              {fieldNewBoxTypes &&
                                 fieldNewBoxTypes.split(',').map(type =>
                                    this.renderBoxTypeCard({
                                       type,
                                       category: 'new',
                                    }),
                                 )}
                           </div>
                        </div>
                     </>
                  )}
               </FormLayout>
            </Form.Section>
            {this.renderButtons()}
         </Form>
      );
   }
}

const validate = values => {
   const errors = {};
   const boxTypes = (values.box_types || '').split(',');
   const newBoxTypes = (values.new_box_types || '').split(',');
   if (values.new_box_type) {
      if (!values.new_box_type.match(/^[A-Za-z0-9-_]+$/)) {
         errors.new_box_type = 'Type must match regexp /[A-Za-z0-9-_]+/';
      }
      if ([...boxTypes, ...newBoxTypes].includes(values.new_box_type)) {
         errors.new_box_type = 'Type must be unique';
      }
   }

   if (!values.temporary_account) {
      if (!values.abc) {
         errors.abc = 'ABC Service is required';
         // eslint-disable-next-line no-underscore-dangle
      } else if (!values.abc._id) {
         errors.abc = 'Select ABC Service from the drop-down list';
      }
   }

   return errors;
};

const selector = formValueSelector('edit-project');

export default compose(
   withNotifications,
   reduxForm({
      form: 'edit-project',
      validate,
      touchOnBlur: false,
      touchOnChange: true,
   }),
   connect(state => ({
      fieldBoxTypes: selector(state, 'box_types'),
      fieldNewBoxTypes: selector(state, 'new_box_types'),
      fieldDeleteBoxTypes: selector(state, 'delete_box_types'),
      fieldNewBoxType: selector(state, 'new_box_type'),
      fieldAbc: selector(state, 'abc'),
      fieldTemporaryAccount: selector(state, 'temporary_account'),
      fieldMonitoringProject: selector(state, 'monitoring_project'),
   })),
)(ProjectEditForm);

function FormFieldTags({ input }) {
   const value = useMemo(() => new Set(input.value), [input.value]);

   const handleChange = useCallback((e, v) => input.onChange(Array.from(v)), [input]);

   return (
      <div data-test={'form-field--tags'}>
         <TagInputEnhanced value={value} onChange={handleChange} />
      </div>
   );
}
