import { ButtonLink, EmptyContainer, EmptyContainerType } from '@yandex-infracloud-ui/libs';
import block from 'bem-cn-lite';
import { Spin } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';

import { urlBuilder } from '../../../models';
import { selectStage } from '../../../redux';
import withNotifications from '../../components/hoc/withNotifications';
import RouteLink from '../../components/RouteLink/RouteLink';
import routes, { createHref } from '../../routes';
import { fetchNamespaces, resetStageNamespaces, selectNamespaces } from '../../store/reducers/balancers';

import './StageBalancers.scss';

const b = block('stage-balancers');

const getEndpointSetsList = project => {
   const { status } = project;
   const deployUnits = Object.keys(status.deploy_units || {});
   const endpointSetsList = [];

   deployUnits.forEach(unitName => {
      const unit = status.deploy_units[unitName];
      const replicaSet = unit.replica_set || unit.multi_cluster_replica_set;

      if (replicaSet && replicaSet.cluster_statuses) {
         const clusterStatuses = replicaSet.cluster_statuses;
         const clusterKeys = Object.keys(clusterStatuses);

         clusterKeys.forEach(cluster => {
            const endpointSetIds = clusterStatuses[cluster].endpoint_set_ids ?? [
               clusterStatuses[cluster].endpoint_set_id,
            ];

            endpointSetIds.forEach(endpointSetId => {
               endpointSetsList.push({ id: endpointSetId, cluster });
            });
         });
      }
   });

   return endpointSetsList;
};

class StageBalancers extends Component {
   static propTypes = {
      fetchNamespaces: PropTypes.func,
      namespaces: PropTypes.array,
      abcServiceId: PropTypes.string,
      resetStageNamespaces: PropTypes.func,
      project: PropTypes.object,
   };

   constructor(props) {
      super(props);

      const { project } = props;
      const endpointSetsList = getEndpointSetsList(project);
      this.hasEndpointSets = endpointSetsList.length > 0;
      this.endpointSets = endpointSetsList;
   }

   componentDidMount() {
      const { hasEndpointSets, endpointSets } = this;
      if (hasEndpointSets) {
         this.props.fetchNamespaces(endpointSets);
      }
   }

   componentWillUnmount() {
      this.props.resetStageNamespaces();
   }

   render() {
      const { namespaces, id } = this.props;
      const { hasEndpointSets } = this;

      let content;

      if (!hasEndpointSets) {
         content = (
            <EmptyContainer
               type={EmptyContainerType.EmptyState}
               title={'No clusters'}
               className={b('empty-container')}
            />
         );
      } else if (namespaces) {
         if (namespaces && namespaces.length) {
            content = namespaces.map(({ namespaceId }) => (
               <RouteLink
                  key={namespaceId}
                  to={createHref(routes.namespaceInfo, { namespaceId })}
                  theme={'normal'}
                  cls={b('link')}
               >
                  {namespaceId}
               </RouteLink>
            ));
         } else {
            content = (
               <EmptyContainer
                  type={EmptyContainerType.EmptyState}
                  title={'No balancers'}
                  className={b('empty-container')}
               />
            );
         }
      }

      return (
         <div className={b()} data-test={'stage-balancers'}>
            {hasEndpointSets && (
               <ButtonLink
                  className={b('new-balancer-button')}
                  size={'m'}
                  theme={'action'}
                  to={urlBuilder.balancerCreate(id)}
               >
                  Launch a new Balancer
               </ButtonLink>
            )}
            {hasEndpointSets && !namespaces ? (
               <div className={b('spinner')}>
                  <Spin size={'l'} progress />
               </div>
            ) : (
               <div className={b('content-container')}>{content}</div>
            )}
         </div>
      );
   }
}

const mapDispatchToProps = dispatch => ({
   fetchNamespaces: endpointSets => fetchNamespaces(dispatch, endpointSets),
   resetStageNamespaces: () => resetStageNamespaces(dispatch),
});

const mapStateToProps = (state, ownProps) => {
   const namespaces = selectNamespaces(state, ownProps.stageId);

   const project = selectStage(state, ownProps.id);
   return {
      namespaces,
      project,
   };
};

export default compose(withRouter, withNotifications, connect(mapStateToProps, mapDispatchToProps))(StageBalancers);
