import { Button } from '@yandex-cloud/uikit';
import {
   DateTimePicker,
   deepClone,
   EmptyContainer,
   EmptyContainerType,
   ExternalLink,
   formatDate,
   isEmpty,
} from '@yandex-infracloud-ui/libs';

import block from 'bem-cn-lite';
import { addHours, addMinutes } from 'date-fns';
import { Spin } from 'lego-on-react';
import PropTypes from 'prop-types';
import React, { Component } from 'react';
import { connect } from 'react-redux';
import { withRouter } from 'react-router-dom';
import { compose } from 'recompose';
import { createStructuredSelector } from 'reselect';

import { getAbsoluteDateFromRelative } from '../../../components/logs-core';
import { LogJson, TimeZonePicker } from '../../../components';
import { LIMIT_OPTIONS } from '../../../components/logs-core/smart-table/constants';
import { StageLogsProvider, StageLogsQueryInput } from '../../../components/query-inputs';
import { EXTERNAL_LINKS, STAGE_LOG_DEFAULT_PERIOD_IN_HOURS, urlBuilder } from '../../../models';
import { handleStopPropagation, setUrlQuery } from '../../../utils';
import CopyToClipboard from '../../components/CopyToClipboard/CopyToClipboard';
import withNotifications from '../../components/hoc/withNotifications';
import Select from '../../components/Select/Select';
import Tooltip from '../../components/Tooltip/Tooltip';
import TooltipInfo from '../../components/TooltipInfo/TooltipInfo';
import { fetchLogs, selectLogs } from '../../store/reducers/deploy';

import { parseQuery } from './queryParser';

import './StageLogs.scss';

const b = block('stage-logs');

const levels = {
   ERROR: 'ERROR',
   WARNING: 'WARNING',
   WARN: 'WARN',
   INFO: 'INFO',
   DEBUG: 'DEBUG',
};

const currentOffset = new Date().getTimezoneOffset(); // Оффсет текущей таймзоны
const DEFAULT_LIMIT = 50;
const LIMIT_VALUES = LIMIT_OPTIONS; // [1, 5, 10, 25, 50, 75, 100];
const NO_SCROLL_FETCH_LIMITS = [1, 5];

const getFetchQueryParams = query => {
   const { errors = [], ...parsedQuery } = parseQuery(query);
   const result = {};
   const keys = {
      'host': 'hostList',
      'pod': 'podList',
      'box': 'boxList',
      'workload': 'workloadList',
      'container_id': 'containerList',
      'logger_name': 'loggerNameList',
      'log_level': 'logLevelList',
      'log_level_int': 'logLevelIntList',
      'pod_transient_fqdn': 'podTransientFqdnList',
      'pod_persistent_fqdn': 'podPersistentFqdnList',
      'node_fqdn': 'nodeFqdnList',
      'thread_name': 'threadNameList',
      'request_id': 'requestIdList',
      'message': 'searchPatternList',
      'stack_trace': 'stackTraceList',
   };

   Object.keys(parsedQuery).forEach(key => {
      if (keys[key]) {
         result[keys[key]] = parsedQuery[key];
      } else if (/^context\./.test(key)) {
         if (!result.userFieldList) {
            result.userFieldList = [];
         }

         result.userFieldList.push({
            ...parsedQuery[key],
            path: key.replace(/^context\./, ''),
         });
      } else {
         errors.push(`'${key}' is unknown parameter. `);
      }
   });

   result.errors = errors;

   return result;
};

const LOCAL_STORAGE_TIMEZONE_KEY = 'LogsTimezoneKey';

class StageLogs extends Component {
   static propTypes = {
      stage: PropTypes.object.isRequired,
      match: PropTypes.object.isRequired,
      location: PropTypes.object.isRequired,
      history: PropTypes.object.isRequired,
      notifications: PropTypes.object.isRequired,
      fetchLogs: PropTypes.func.isRequired,
      selectLogs: PropTypes.func.isRequired,
      logs: PropTypes.object,
   };

   constructor(props) {
      super(props);

      const timezoneKey = window.localStorage.getItem(LOCAL_STORAGE_TIMEZONE_KEY);
      let timezoneData;
      if (timezoneKey) {
         timezoneData = TimeZonePicker.GetTimezoneByKey(timezoneKey);
      } else {
         timezoneData = TimeZonePicker.GetCurrentTimezone();
      }

      const offsetDelta = timezoneData.offset - currentOffset;
      const now = new Date();
      const dateFrom = addHours(addMinutes(now, offsetDelta), -STAGE_LOG_DEFAULT_PERIOD_IN_HOURS);

      this.state = {
         filters: {
            query: '',
            limit: DEFAULT_LIMIT,
            order: 'DESC',
            dateFrom,
         },
         query: '',
         navigation: {
            activeList: {},
         },
         suggestProvider: new StageLogsProvider({}),
         timezoneData,
      };

      /**
       * Флажок-костыль, для обхода проблемы двух источников истины (фильтров: из URL и из state)
       *
       * При браузерной навигации он остается false, что вызывает перезагрузку данных в componentDidUpdate.
       *
       * При обновлении URL пользователем, путем редактирования фильтров, перезагрузка данных происходит явно,
       * поэтому повторная перезагрузка в componentDidUpdate подавляется.
       */
      this.preventReloadDataOnNextUpdate = false;
   }

   static getDerivedStateFromProps(props, state) {
      const { stage } = props;

      state.filters.projectId = stage?.meta?.project_id;

      if (!state.filters.deployUnitId && stage?.spec?.deploy_units) {
         const deployUnits = Object.keys(stage.spec.deploy_units);

         if (deployUnits[0]) {
            // eslint-disable-next-line prefer-destructuring
            state.filters.deployUnitId = deployUnits[0];
         }
      }

      return state;
   }

   componentDidMount() {
      const search = new URLSearchParams(this.props.location.search);
      const continuationToken = search.has('range') ? search.get('range') : null;

      if (search.has('highlight')) {
         const highlight = Number(search.get('highlight'));

         if (highlight > 0) {
            const waitForElementToDisplay = (selector, time) => {
               const rowFrom = document.querySelector('[data-scroll=log-0]');
               const rowTo = document.querySelector(selector);

               if (rowFrom && rowTo) {
                  const fromY = rowFrom.getBoundingClientRect().top;
                  const toY = rowTo.getBoundingClientRect().top;
                  window.scrollTo({
                     top: window.scrollY + toY - fromY,
                     left: 0,
                     behavior: 'auto', // smooth/auto
                  });
               } else {
                  setTimeout(() => {
                     waitForElementToDisplay(selector, time);
                  }, time);
               }
            };

            waitForElementToDisplay(`[data-scroll=log-${highlight - 1}]`, 1000);

            this.setState(state => ({
               navigation: {
                  ...state.navigation,
                  activeList: {
                     [`log-${highlight - 1}`]: { highlighted: true },
                  },
               },
            }));
         }
      }

      const filters = this.getFiltersFromUrl(this.props.location.search);

      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               ...filters,
            },
            query: filters.query || '',
         }),
         () => {
            this.fetchLogsFromStateParams(continuationToken ? { continuationToken } : null);
            this.updateSuggestProvider();
         },
      );

      window.addEventListener('scroll', this.onTableScroll);
   }

   componentDidUpdate(prevProps, prevState, snapshot) {
      // Отработка смены URL (браузерная навигация)
      if (this.props.location.search !== prevProps.location.search) {
         if (this.preventReloadDataOnNextUpdate) {
            this.preventReloadDataOnNextUpdate = false;
         } else {
            const filters = this.getFiltersFromUrl(this.props.location.search);

            this.setState(
               {
                  filters,
                  query: filters.query ?? '',
               },
               () => this.reloadData(),
            );
         }
      }
   }

   componentWillUnmount() {
      const { notifications } = this.props;

      Object.keys(notifications.notificationsData).forEach(notificationId => {
         notifications.remove(notificationId);
      });

      window.removeEventListener('scroll', this.onTableScroll);
   }

   onTableScroll = () => {
      const { logs } = this.props;
      const {
         fetchedToken,
         filters: { deployUnitId, limit },
      } = this.state;

      if (NO_SCROLL_FETCH_LIMITS.includes(limit)) {
         return;
      }

      const loadMore = document.querySelector('[data-load="more"]');

      const filteredLogs = logs && logs[deployUnitId] ? logs[deployUnitId] : {};
      const { continuationTokens } = filteredLogs;

      if (loadMore) {
         const { bottom } = loadMore.getBoundingClientRect();

         if (!this.props.fetching && bottom - window.innerHeight < 200) {
            if (continuationTokens && continuationTokens.forward) {
               if (!fetchedToken || fetchedToken !== continuationTokens.forward) {
                  this.setState(
                     {
                        fetchedToken: continuationTokens.forward,
                     },
                     () => {
                        this.fetchLogsFromStateParams({ continuationToken: continuationTokens.forward });
                     },
                  );
               }
            }
         }
      }
   };

   onFiltersSubmit = () => {
      this.updateURLFromStateParams();
      this.reloadData();
   };

   onFilterDeployUnitChange = deployUnitId => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               deployUnitId: deployUnitId[0],
               query: prevState.query,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onFilterLevelChange = level => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               level,
               query: prevState.query,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onFiltersDateFromChange = (event, dateFrom) => {
      const { timezoneData } = this.state;

      const offsetDelta = timezoneData.offset - currentOffset;
      const dateFromWithoutTimezone = dateFrom ? new Date(dateFrom.getTime() + offsetDelta * 60 * 1000) : null;

      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               dateFrom: dateFromWithoutTimezone,
               query: prevState.query,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onFiltersDateToChange = (event, dateTo) => {
      const { timezoneData } = this.state;

      const offsetDelta = timezoneData.offset - currentOffset;
      const dateToWithoutTimezone = dateTo ? new Date(dateTo.getTime() + offsetDelta * 60 * 1000) : null;

      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               dateTo: dateToWithoutTimezone,
               query: prevState.query,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onOrderChange = () => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               order: prevState.filters.order === 'DESC' ? 'ASC' : 'DESC',
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onFiltersReset = () => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               dateFrom: null,
               dateTo: null,
               level: null,
               order: 'DESC',
               query: '',
               limit: DEFAULT_LIMIT,
            },
            query: '',
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onSuggestChange(value) {
      this.setState({
         query: value,
      });
   }

   onSearchClick = () => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               query: prevState.query,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onSuggestLinkClick = value => {
      if (value === 'message' || value === 'stack_trace') {
         this.setState({
            query: `${value} = "Search string with \\"quotes\\"", "Search string with other punctuation characters (=,;!)"; `,
         });
      } else {
         this.setState({
            query: `${value} = ${value}_1, ${value}_2, ${value}_3; `,
         });
      }
   };

   onLogRowClick(log, logIndex) {
      const { activeList } = this.state.navigation;

      if (logIndex && log.timestamp) {
         if (
            activeList &&
            activeList[
               logIndex
            ] /* && activeList[logIndex].timestamp && activeList[logIndex].timestamp === log.timestamp */
         ) {
            activeList[logIndex] = undefined;
         } else {
            activeList[logIndex] = { timestamp: log.timestamp };
         }

         this.setState(state => ({
            navigation: {
               ...state.navigation,
               activeList: {
                  ...activeList,
               },
            },
         }));
      }
   }

   onJsonValueClick = (field, value, path) => {
      const { query } = this.state;
      const { errors, ...parsedQuery } = parseQuery(query);

      if (!query || !errors || !errors.length) {
         if (path && path[0]) {
            if (path[0] === 'context') {
               const contextPath = path.map(v => v.replaceAll('.', '\\.')).join('.'); // DEPLOY-5864

               if (!parsedQuery[contextPath] || parsedQuery[contextPath].select_type === 'EXCLUDE') {
                  parsedQuery[contextPath] = {
                     select_type: 'INCLUDE',
                     values: [value],
                  };
               } else if (!parsedQuery[contextPath].values.includes(`${value}`)) {
                  parsedQuery[contextPath] = {
                     select_type: 'INCLUDE',
                     values: [...parsedQuery[contextPath].values, value],
                  };
               }
            } else if (!parsedQuery[field] || parsedQuery[field].select_type === 'EXCLUDE') {
               parsedQuery[field] = {
                  select_type: 'INCLUDE',
                  values: [value],
               };
            } else if (!parsedQuery[field].values.includes(`${value}`)) {
               parsedQuery[field].values = [...parsedQuery[field].values, value];
            }
         }

         let newQuery = '';

         if (parsedQuery && Object.keys(parsedQuery).length) {
            Object.keys(parsedQuery).forEach(key => {
               const formatValue = v =>
                  typeof v === 'string' && (v === '' || /["\s,!=;]/.test(v)) ? `"${v.replace(/"/g, '\\"')}"` : v;

               newQuery += key;
               newQuery += parsedQuery[key].select_type === 'INCLUDE' ? '=' : '!=';
               newQuery += parsedQuery[key].values.map(formatValue).join(', ');
               newQuery += '; ';
            });
         }

         this.onSuggestChange(newQuery);
      }
   };

   onTimezoneSet = timezoneData => {
      this.setState({ timezoneData });
      window.localStorage.setItem(LOCAL_STORAGE_TIMEZONE_KEY, timezoneData.key);
   };

   onLimitChange = ([limit]) => {
      this.setState(
         prevState => ({
            filters: {
               ...prevState.filters,
               query: prevState.query,
               limit,
            },
         }),
         () => this.onFiltersSubmit(),
      );
   };

   onLoadMore = () => {
      const { logs } = this.props;
      const {
         fetchedToken,
         filters: { deployUnitId },
      } = this.state;
      const filteredLogs = logs && logs[deployUnitId] ? logs[deployUnitId] : {};
      const { continuationTokens } = filteredLogs;

      if (!fetchedToken || fetchedToken !== continuationTokens.forward) {
         this.setState(
            {
               fetchedToken: continuationTokens.forward,
            },
            () => {
               this.fetchLogsFromStateParams({
                  continuationToken: continuationTokens.forward,
               });
            },
         );
      }
   };

   getFiltersFromUrl(rawSearch) {
      const search = new URLSearchParams(rawSearch);
      const filters = {};

      if (search.has('deploy-unit')) {
         filters.deployUnitId = search.get('deploy-unit');
      } else if (search.has('deployUnitId')) {
         filters.deployUnitId = search.get('deployUnitId');
      }

      if (search.has('order')) {
         filters.order = search.get('order') === 'ASC' ? 'ASC' : 'DESC';
      }

      if (search.has('date-from')) {
         filters.dateFrom = new Date(search.get('date-from'));
      } else if (search.has('from')) {
         const absolute = Number(search.get('from'));

         filters.dateFrom = absolute ? new Date(absolute * 1000) : getAbsoluteDateFromRelative(search.get('from'));
      }

      if (search.has('date-to')) {
         filters.dateTo = new Date(search.get('date-to'));
      } else if (search.has('to')) {
         const absolute = Number(search.get('to'));

         filters.dateTo = absolute ? new Date(absolute * 1000) : getAbsoluteDateFromRelative(search.get('to'));
      }

      if (search.has('level') && search.get('level').length) {
         filters.level = search.get('level').split(',');
      }

      if (search.has('query')) {
         filters.query = search.get('query');
      }

      if (search.has('limit')) {
         const limit = parseInt(search.get('limit'), 10);
         filters.limit = Number.isNaN(limit) ? DEFAULT_LIMIT : limit;
      }

      return filters;
   }

   getUrlParamsFromState() {
      const {
         filters: { deployUnitId, dateFrom, dateTo, order, query, level, limit },
      } = this.state;
      const search = {};

      if (deployUnitId) {
         // search['deploy-unit'] = deployUnitId;
         search.deployUnitId = deployUnitId;
      }

      if (order && order === 'ASC') {
         search.order = order;
      }

      if (level && level.length) {
         search.level = level;
      }

      if (dateFrom) {
         // search['date-from'] = dateFrom.toISOString();
         search.from = Math.round(dateFrom.getTime() / 1000);
      }

      if (dateTo) {
         // search['date-to'] = dateTo.toISOString();
         search.to = Math.round(dateTo.getTime() / 1000);
      }

      if (query && query.length) {
         search.query = query;
      }

      if (limit && limit !== DEFAULT_LIMIT) {
         search.limit = limit;
      }

      return search;
   }

   fetchLogsFromStateParams(customParams) {
      const { match } = this.props;
      const { filters } = this.state;
      const { projectId, deployUnitId, query, limit, order, dateFrom, dateTo, level } = this.state.filters;
      const { errors, ...parsedQueryParams } = getFetchQueryParams(query);

      if (!(errors && errors.length)) {
         const timestampRange = {};

         if (dateFrom) {
            timestampRange.begin = dateFrom;
         }

         if (dateTo) {
            timestampRange.end = dateTo;
         }

         if (level && level.length) {
            if (!parsedQueryParams.logLevelList) {
               parsedQueryParams.logLevelList = {
                  values: level.includes(levels.WARNING) ? [...level, levels.WARN] : level,
                  select_type: 'INCLUDE',
               };
            }
         }

         this.props.fetchLogs({
            params: {
               projectId,
               stageId: match.params.stageId,
               deployUnitId,
               timestampRange,
               ...parsedQueryParams,
               ...customParams,
               limit,
               order,
               filters,
            },
         });
      }
   }

   reloadData() {
      this.setState(
         prevState => ({
            navigation: {
               ...prevState.navigation,
               activeList: {},
            },
         }),
         () => {
            this.fetchLogsFromStateParams();
            this.updateSuggestProvider();
         },
      );
   }

   updateSuggestProvider() {
      const { filters } = this.state;
      const { match } = this.props;

      const timestampRange = {};

      if (filters.dateFrom) {
         timestampRange.begin = filters.dateFrom;
      }

      if (filters.dateTo) {
         timestampRange.end = filters.dateTo;
      }

      const suggestProvider = new StageLogsProvider({
         deployUnitId: filters.deployUnitId,
         order: filters.order,
         projectId: filters.projectId,
         stageId: match.params.stageId,
         timestampRange,
      });

      this.setState({
         suggestProvider,
      });
   }

   updateURLFromStateParams() {
      const { history, location } = this.props;
      const search = this.getUrlParamsFromState();

      this.preventReloadDataOnNextUpdate = true;
      setUrlQuery(history, location, search, true);
   }

   renderJsonValueComponent = ({ field, value, path, children }) => {
      const jsonClickableFieldList = [
         'host',
         'pod',
         'pod_transient_fqdn',
         'pod_persistent_fqdn',
         'node_fqdn',
         'container_id',
         'box',
         'workload',
         'logger_name',
         'log_level',
         'log_level_int',
         'thread_name',
         'request_id',
         // 'message',
         // 'stack_trace',
         'context',
      ];

      if (path && path[0] && jsonClickableFieldList.includes(path[0])) {
         return (
            <span
               className={'clickable'}
               onClick={() => {
                  this.onJsonValueClick(field, value, path);
               }}
               tabIndex={0}
               aria-label={'Json value'}
               role={'button'}
            >
               {children}
            </span>
         );
      }

      return children;
   };

   renderLogsFilters() {
      const {
         query,
         filters: { deployUnitId, level, limit, dateFrom, dateTo },
         timezoneData,
      } = this.state;
      const { errors, ...parsedQueryParams } = getFetchQueryParams(query);

      const { stage } = this.props;
      const deployUnits = Object.keys(stage.spec.deploy_units);

      const offsetDelta = timezoneData.offset - currentOffset;

      const dateFromInTimezone = dateFrom ? new Date(dateFrom.getTime() - offsetDelta * 60 * 1000) : null;
      const dateToInTimezone = dateTo ? new Date(dateTo.getTime() - offsetDelta * 60 * 1000) : null;

      const hasErrors = errors && errors.length > 0;

      return (
         <div className={b('filters')}>
            <div className={b('filter-items-container')}>
               <div className={b('filters-items', { controls: true })}>
                  <div className={b('filter-row', { filters: true })}>
                     {deployUnits.length ? (
                        <div
                           className={b('filter', { type: 'deploy-unit' }, b('filter-item'))}
                           data-test={'filter-logs-by-deploy-unit'}
                        >
                           <Select
                              val={deployUnitId}
                              button={{
                                 text: deployUnitId || 'Choose Deploy Unit',
                              }}
                              width={'max'}
                              items={deployUnits.sort().map(deployUnit => ({
                                 key: `filter-deploy-unit-${deployUnit}`,
                                 val: deployUnit,
                                 text: deployUnit,
                              }))}
                              onChange={this.onFilterDeployUnitChange}
                              readonly={!deployUnits || !deployUnits.length || deployUnits.length === 1}
                           />
                        </div>
                     ) : (
                        <Spin size={'xxs'} progress />
                     )}
                     {
                        <div
                           className={b('filter', { type: 'level' }, b('filter-item'))}
                           data-test={'filter-logs-by-level'}
                        >
                           <Select
                              type={'check'}
                              val={level || null}
                              placeholder={'Log levels'}
                              button={{
                                 text: level && level.length ? level.join(', ') : 'any level',
                              }}
                              width={'max'}
                              items={[levels.ERROR, levels.WARNING, levels.INFO, levels.DEBUG].map(value => ({
                                 key: `key--filter-level-${value}`,
                                 val: value,
                                 text: value,
                              }))}
                              onChange={this.onFilterLevelChange}
                              readonly={parsedQueryParams.logLevelList !== undefined}
                           />
                        </div>
                     }
                     <div className={b('filter-dates-wrapper', null, b('filter-item'))}>
                        <div className={b('filter', { type: 'date' })} data-test={'filter-logs-from-date'}>
                           <DateTimePicker
                              value={dateFromInTimezone}
                              withTime={true}
                              showTimeString={true}
                              onChange={this.onFiltersDateFromChange}
                              withSeconds={true}
                           />
                        </div>
                        <div className={b('filter-label')}>—</div>
                        <div className={b('filter', { type: 'date' })} data-test={'filter-logs-to-date'}>
                           <DateTimePicker
                              value={dateToInTimezone}
                              withTime={true}
                              showTimeString={true}
                              onChange={this.onFiltersDateToChange}
                              withSeconds={true}
                           />
                        </div>
                     </div>
                     <div className={b('filter', { type: 'timezone' }, b('filter-item'))} data-test={'set-timezone'}>
                        <span className={b('timezone-title')}>Timezone</span>
                        <TimeZonePicker timeZoneKey={timezoneData.key} onChange={this.onTimezoneSet} />
                     </div>
                     <div className={b('filter', { type: 'limit' }, b('filter-item'))} data-test={'set-limit'}>
                        <span className={b('filter-title')}>Limit</span>
                        <Select
                           val={limit}
                           items={LIMIT_VALUES.map(l => ({ val: l, text: l }))}
                           onChange={this.onLimitChange}
                        />
                     </div>
                  </div>
               </div>
               <div className={b('filters-items', { buttons: true })}>
                  <div className={b('filter-reset')} data-test={'filter-reset'}>
                     <Button view={'outlined'} width={'max'} onClick={this.onFiltersReset}>
                        Reset
                     </Button>
                  </div>
               </div>
            </div>
            <div className={b('filter-items-container')}>
               <div className={b('filters-items', { controls: true })}>
                  <div className={b('filter-row', { query: true })}>
                     <div className={b('filter-query')}>
                        <StageLogsQueryInput
                           query={this.state.query || ''}
                           provider={this.state.suggestProvider}
                           onChange={value => this.onSuggestChange(value)}
                        />
                     </div>
                  </div>
               </div>
               <div className={b('filters-items', { buttons: true })}>
                  <div className={b('filter-submit')} data-test={'filter-logs-submit'}>
                     <Button view={'action'} width={'max'} disabled={hasErrors} onClick={this.onSearchClick}>
                        Search
                     </Button>
                  </div>
               </div>
            </div>
            <div className={b('filter-row', { help: true })}>
               {hasErrors && <span className={b('error')}>{errors.join(' ')}</span>}
               {'Search by keys: '}
               {['host', 'pod', 'box', 'workload', 'container_id', 'logger_name', 'message', 'stack_trace'].map(
                  (field, index) => (
                     <React.Fragment key={`key--suggest-link-${field}`}>
                        {index > 0 ? ', ' : ''}
                        <span
                           className={b('link')}
                           onClick={() => this.onSuggestLinkClick(field)}
                           role={'link'}
                           tabIndex={0}
                        >
                           {field}
                        </span>
                     </React.Fragment>
                  ),
               )}
               {'. '}
               {<TooltipInfo note={'logsFilter'} />}
            </div>
         </div>
      );
   }

   renderLogLink(log, logIndex = undefined) {
      const { stageId } = this.props.match.params;
      const url = `https://${window.location.host}`;
      const search = this.getUrlParamsFromState();

      if (log) {
         if (log.linkToken) {
            search.range = log.linkToken;

            if (logIndex !== undefined) {
               search.highlight = logIndex + 1;
            }
         }
      }

      return (
         <div className={b('copy-link')}>
            <CopyToClipboard
               value={url + urlBuilder.stageLogs(stageId, search)}
               icon={'link'}
               onClick={handleStopPropagation}
            />
         </div>
      );
   }

   renderCellLogMessageOpened(logSource) {
      let jsonContext = null;
      let jsonMessage = null;

      // чтобы не модифицировать при каждом расхлопе исходный объект
      const log = deepClone(logSource);

      if (log.log.context !== undefined) {
         try {
            jsonContext = JSON.parse(log.log.context);

            if (jsonContext) {
               // меняем на json, если парсится как json
               log.log.context = jsonContext;
            }
         } catch {
            console.log(`"context" isn't a JSON:\n${log.log.context}`);
         }
      }

      if (log.log.message !== undefined) {
         try {
            jsonMessage = JSON.parse(log.log.message);

            if (jsonMessage) {
               // меняем на json, если парсится как json
               log.log.message = jsonMessage;
            }
         } catch {
            console.log(`"message" isn't a JSON:\n${log.log.message}`);
         }
      } else if (log.log.log_message !== undefined) {
         // @nodejsgirl кажется, что уже можно оторвать этот кусок
         // TODO: DEPLOY-2318
         // AWAIT: DEPLOY-2589
         try {
            jsonMessage = JSON.parse(log.log.log_message);

            if (jsonMessage) {
               log.log.log_message = jsonMessage;
            }
         } catch {
            console.log(`"log_message" isn't a JSON:\n${log.log.log_message}`);
         }
      }

      return (
         <div className={b('log-opened')}>
            <div className={`${b('log-opened-source')}`}>{this.renderLogSourceInfo(log)}</div>
            <div className={`${b('log-opened-log-json')}`}>
               <LogJson log={log.log} customValueComponent={this.renderJsonValueComponent} />
            </div>
            <div className={`${b('log-opened-copy')}`}>
               {/* бывают какие-то конские логи с message под миллион символов и на сто экранов, которые ломают в этом месте UI #DEPLOY-4988 */}
               {/* проверяем длину logMessage, т.к. это всегда строковое значение, а message может попарситься в джейсон */}
               {!isEmpty(log.logMessage) && log.logMessage.length < 99999 ? (
                  <CopyToClipboard value={JSON.stringify(log.log, null, '  ')} onClick={handleStopPropagation} />
               ) : null}
            </div>
         </div>
      );
   }

   renderCellLogTime(log) {
      const { timestamp } = log;
      const { timezoneData } = this.state;

      const offsetDelta = timezoneData.offset - currentOffset;
      const timezoneTimestamp = new Date(timestamp).getTime() - offsetDelta * 60 * 1000;

      return (
         <>
            <div className={b('log-timestamp', { date: true })} data-test={'log--timestamp-date'}>
               {formatDate(timezoneTimestamp, 'd MMM yyyy')}
            </div>
            <div className={b('log-timestamp', { time: true })} data-test={'log--timestamp-time'}>
               {formatDate(timezoneTimestamp, 'HH:mm:ss.SSS')}{' '}
               <span className={b('log-data-timezone')}>({timezoneData.title})</span>
            </div>
         </>
      );
   }

   renderCellLogLevel(log) {
      return (
         <div
            className={b('log-level', {
               error: log.logLevel === levels.ERROR,
               warning: log.logLevel === levels.WARNING || log.logLevel === levels.WARN,
               info: log.logLevel === levels.INFO,
               debug: log.logLevel === levels.DEBUG,
            })}
            title={log.logLevel}
         >
            {log.logLevel === levels.ERROR && <i className={'far fa-exclamation-triangle'} />}
            {log.logLevel === levels.WARNING ||
               (log.logLevel === levels.WARN && <i className={'far fa-exclamation-triangle'} />)}
            {log.logLevel === levels.INFO && <i className={'far fa-info-square'} />}
            {log.logLevel === levels.DEBUG && <i className={'far far fa-cog'} />}
            {` ${log.logLevel || '—'}`}
         </div>
      );
   }

   renderCellLogMessage(log) {
      const message = log.logMessage.replace(/\n/, ' ') || '';

      return (
         <div className={b('log-message-short')}>
            {message.length > 300 ? (
               <>
                  {`${message.slice(0, 250)}...`}
                  {/* для "конских" логов #DEPLOY-4988 */}{' '}
                  <span className={b('skipped-text')}>(+{log.log.message.length - 250})</span>
               </>
            ) : (
               message
            )}
         </div>
      );
   }

   renderCellLogSource(log) {
      return (
         <Tooltip mix={'log-source'} text={this.renderLogSourceInfo(log)} directions={['bottom-right', 'top-right']}>
            {`... / ${log.workload}`}
         </Tooltip>
      );
   }

   renderLogSourceInfo(log) {
      const {
         filters: { deployUnitId },
      } = this.state;

      return (
         <>
            <span className={b('log-source-popup-deploy-unit')}>{deployUnitId}</span>
            <span className={b('log-source-popup-box')}>{log.box}</span>
            <span className={b('log-source-popup-workload')}>{log.workload}</span>
         </>
      );
   }

   renderCellLogLogger(log) {
      return log.loggerName?.length > 10 ? (
         <Tooltip mix={'log-logger'} text={log.loggerName} directions={['bottom-right', 'top-right']}>
            <div className={b('log-logger-tooltip')}>{log.loggerName}</div>
         </Tooltip>
      ) : (
         log.loggerName
      );
   }

   renderLogsTable() {
      const { logs, stage } = this.props;

      const {
         fetchedToken,
         filters: { deployUnitId, limit, order, projectId, query },
         navigation: { activeList },
      } = this.state;

      const filteredLogs = logs && logs[deployUnitId] ? logs[deployUnitId] : {};
      const { errors } = getFetchQueryParams(query);

      if (errors && errors.length) {
         return null;
      }

      const { error, continuationTokens } = filteredLogs;

      if (error) {
         return (
            <div className={b('something-went-wrong')}>
               <div className={b('error')}>{error.message ? `${error.message}` : 'Something went wrong... '}</div>
            </div>
         );
      }

      return filteredLogs && filteredLogs.logEntries && filteredLogs.filters === this.state.filters ? (
         <>
            {filteredLogs.logEntries.length ? (
               <>
                  <table className={b('table')}>
                     <thead>
                        <tr data-scroll={'thead'}>
                           <th className={b('th')}>
                              <div
                                 className={b('th__date-sort')}
                                 onClick={this.onOrderChange}
                                 role={'button'}
                                 aria-label={'Sort'}
                                 tabIndex={0}
                              >
                                 Date
                                 {order === 'ASC' ? (
                                    <i className={'far fa-sort-amount-down-alt'} />
                                 ) : (
                                    <i className={'far fa-sort-amount-down'} />
                                 )}
                              </div>
                           </th>
                           <th className={b('th')} aria-label={'Links'} />
                           <th className={b('th')}>Level</th>
                           <th className={b('th')}>Message</th>
                           <th className={b('th')}>Source</th>
                           <th className={b('th')}>Logger</th>
                        </tr>
                     </thead>
                     <tbody className={b('tbody')}>
                        {filteredLogs.logEntries.map((log, logIndex) => {
                           const isActive =
                              activeList[`log-${logIndex}`] &&
                              activeList[`log-${logIndex}`].timestamp &&
                              activeList[`log-${logIndex}`].timestamp === log.timestamp;
                           const isHighlighted =
                              activeList[`log-${logIndex}`] && activeList[`log-${logIndex}`].highlighted;

                           return (
                              <React.Fragment key={`log-${log.timestamp}`}>
                                 {logIndex > 0 && logIndex % limit === 0 && limit > 1 ? (
                                    <tr className={b('tr')}>
                                       <td className={b('td', { range: true })}>
                                          {logIndex + 1} ... {logIndex + limit}
                                       </td>
                                       <td colSpan={5} className={b('td', { range: true })}>
                                          {this.renderLogLink(log)}
                                       </td>
                                    </tr>
                                 ) : null}
                                 <tr
                                    data-scroll={`log-${logIndex}`}
                                    className={b('tr', {
                                       'log-row': true,
                                       hoverable: true,
                                       'active-row': isActive || isHighlighted,
                                       'highlighted-row': isHighlighted,
                                    })}
                                    onClick={this.onLogRowClick.bind(this, log, `log-${logIndex}`)}
                                    data-test={'logs--row'}
                                 >
                                    <td className={b('td', { 'log-time': true })}>
                                       {isActive || isHighlighted ? (
                                          <div className={b('row-icon', { collapse: true })}>
                                             <i className={'far fa-angle-up'} />
                                          </div>
                                       ) : (
                                          <div className={b('row-icon', { expand: true })}>
                                             <i className={'far fa-angle-down'} />
                                          </div>
                                       )}
                                       {this.renderCellLogTime(log)}
                                    </td>
                                    {/* eslint-disable-next-line jsx-a11y/no-noninteractive-element-interactions */}
                                    <td className={b('td', { 'log-link': true })} onClick={handleStopPropagation}>
                                       {this.renderLogLink(log, logIndex % limit)}
                                    </td>
                                    <td className={b('td', { 'log-level': true })}>{this.renderCellLogLevel(log)}</td>
                                    <td className={b('td', { 'log-message': true })} data-test={'log--message'}>
                                       {this.renderCellLogMessage(log)}
                                    </td>
                                    <td className={b('td', { 'log-source': true })} data-test={'log--source'}>
                                       {this.renderCellLogSource(log)}
                                    </td>
                                    <td className={b('td', { 'log-logger': true })} data-test={'log--logger'}>
                                       {this.renderCellLogLogger(log)}
                                    </td>
                                 </tr>
                                 {isActive || isHighlighted ? (
                                    <tr
                                       className={b('tr', { 'opened-row': true, 'highlighted-row': isHighlighted })}
                                       data-test={'log--expand'}
                                    >
                                       <td colSpan={6} className={b('td', { 'log-opened': true })}>
                                          {this.renderCellLogMessageOpened(log)}
                                       </td>
                                    </tr>
                                 ) : null}
                              </React.Fragment>
                           );
                        })}
                     </tbody>
                  </table>
                  <div className={b('load-more')} data-test={'log--load-more'} data-load={'more'}>
                     {continuationTokens && continuationTokens.forward ? (
                        <>
                           {fetchedToken && fetchedToken === continuationTokens.forward ? (
                              <div className={b('loading-more-logs')}>
                                 <Spin size={'xxs'} progress />
                              </div>
                           ) : (
                              <Button width={'max'} onClick={this.onLoadMore}>
                                 Load more entries
                              </Button>
                           )}
                        </>
                     ) : (
                        <div className={b('no-more-logs')}>no more logs</div>
                     )}
                  </div>
               </>
            ) : (
               <div data-test={'logs-not-found'}>
                  <EmptyContainer
                     // TODO: DEPLOY-812
                     // EmptyContainerType.EmptyState, если логи выключены
                     // EmptyContainerType.NotFound, если логи включены, но не найдены
                     type={EmptyContainerType.EmptyState}
                     title={'There are no results'}
                     description={
                        <>
                           Try to change a search query or filters.
                           <div>
                              <ExternalLink href={EXTERNAL_LINKS.yqlStageLogsQuery(stage.meta.id, projectId)}>
                                 Search for logs in YT
                              </ExternalLink>
                           </div>
                        </>
                     }
                  />
               </div>
            )}
         </>
      ) : (
         <div className={b('spinner')}>
            <Spin size={'s'} progress />
         </div>
      );
   }

   render() {
      const { stage } = this.props;
      const deployUnits = stage?.spec?.deploy_units ? Object.keys(stage.spec.deploy_units) : [];

      if (deployUnits.length === 0) {
         return (
            <div className={b('no-deploy-units')} data-test={'no-deploy-units'}>
               Please define at least one Deploy Unit
            </div>
         );
      }

      return (
         <div className={b()} data-test={'view-stage--logs'}>
            {this.renderLogsFilters()}
            {this.renderLogsTable()}
         </div>
      );
   }
}

const mapStateToProps = createStructuredSelector({
   logs: (state, ownProps) => selectLogs(state, ownProps.match.params.stageId),
});

const mapDispatchToProps = {
   fetchLogs,
   selectLogs,
};

export default compose(withRouter, withNotifications, connect(mapStateToProps, mapDispatchToProps))(StageLogs);
