import { parseQuery } from './queryParser';

it('should parse correct query', () => {
   const parsedQuery = parseQuery(
      '  log_level=level1,level2; logger_name!=logger1,logger2; message="HTTP/1.0\\" ", "escaping queries \\"1\\" \\"2\\" \\"3\\"        "',
   );

   expect(parsedQuery).toEqual({
      errors: [],
      log_level: {
         select_type: 'INCLUDE',
         values: ['level1', 'level2'],
      },
      logger_name: {
         select_type: 'EXCLUDE',
         values: ['logger1', 'logger2'],
      },
      message: {
         select_type: 'INCLUDE',
         values: ['HTTP/1.0" ', 'escaping queries "1" "2" "3"        '],
      },
   });
});

it('should parse context queries', () => {
   const parsedQuery = parseQuery(
      ' context.context.with space = "space"; context.context.with\\.dot != "dot"; context.context.with-minus = "minus";  ',
   );

   expect(parsedQuery).toEqual({
      errors: [],
      'context.context.with space': {
         select_type: 'INCLUDE',
         values: ['space'],
      },
      'context.context.with\\.dot': {
         select_type: 'EXCLUDE',
         values: ['dot'],
      },
      'context.context.with-minus': {
         select_type: 'INCLUDE',
         values: ['minus'],
      },
   });
});

it('should parse incorrect queries', () => {
   // incorrect queries:
   // message ==!!== abc
   // mess age = a b
   // message a b
   // message = a b
   // message = "a ;
   // message = b " c "
   // message = " c " d
   // message = " d " " e "
   // message = e , ""  f  ""
   // message = , , ,

   const queries = [
      {
         query: 'message ==!!== abc',
         errors: ['Search query is incorrect. ', 'Syntax error: "==!!==". '],
      },
      {
         query: 'mess age = a b',
         errors: ['Search query is incorrect. ', 'Key error (mess age). '],
      },
      {
         query: 'message a b',
         errors: ['Search query is incorrect. ', 'Key error (message a b). '],
      },
      {
         query: 'message = a b',
         errors: ['Search query is incorrect. ', 'Values error (a b). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = "a ;',
         errors: ['Search query is incorrect. ', 'Values error (a ;). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = b " c "',
         errors: ['Search query is incorrect. ', 'Values error (b "). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = " c " d',
         errors: ['Search query is incorrect. ', 'Values error (" c " d). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = " d " " e "',
         errors: ['Search query is incorrect. ', 'Values error ( d "). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = e , ""  f  ""',
         errors: ['Search query is incorrect. ', 'Values error (" " f). ', 'Commas or double quotes are required. '],
      },
      {
         query: 'message = , , ,',
         errors: ['Search query is incorrect. ', 'Value error. '],
      },
   ];

   queries.forEach(item => {
      expect(parseQuery(item.query)).toEqual({
         errors: item.errors,
      });
   });
});
