import createReactClass from 'create-react-class';
import PropTypes from 'prop-types';
import React from 'react';
import Constants from './constants';
import NotificationContainer from './NotificationContainer';
import Styles from './styles';

var NotificationSystem = createReactClass({
   uid: 3400,

   _isMounted: false,

   _getStyles: {
      overrideStyle: {},

      overrideWidth: null,

      setOverrideStyle: function (style) {
         this.overrideStyle = style;
      },

      wrapper: function () {
         if (!this.overrideStyle) return {};
         return Object.assign({}, Styles.Wrapper, this.overrideStyle.Wrapper);
      },

      container: function (position) {
         var override = this.overrideStyle.Containers || {};
         if (!this.overrideStyle) return {};

         this.overrideWidth = Styles.Containers.DefaultStyle.width;

         if (override.DefaultStyle && override.DefaultStyle.width) {
            this.overrideWidth = override.DefaultStyle.width;
         }

         if (override[position] && override[position].width) {
            this.overrideWidth = override[position].width;
         }

         return Object.assign(
            {},
            Styles.Containers.DefaultStyle,
            Styles.Containers[position],
            override.DefaultStyle,
            override[position],
         );
      },

      elements: {
         notification: 'NotificationItem',
         title: 'Title',
         messageWrapper: 'MessageWrapper',
         dismiss: 'Dismiss',
         action: 'Action',
         actionWrapper: 'ActionWrapper',
      },

      byElement: function (element) {
         var self = this;
         return function (level) {
            var _element = self.elements[element];
            var override = self.overrideStyle[_element] || {};
            if (!self.overrideStyle) return {};
            return Object.assign(
               {},
               Styles[_element].DefaultStyle,
               Styles[_element][level],
               override.DefaultStyle,
               override[level],
            );
         };
      },
   },

   _didNotificationRemoved: function (uid) {
      var notification;
      var notifications = this.state.notifications.filter(function (toCheck) {
         if (toCheck.uid === uid) {
            notification = toCheck;
            return false;
         }
         return true;
      });

      if (this._isMounted) {
         this.setState({ notifications: notifications });
      }

      if (notification && notification.onRemove) {
         notification.onRemove(notification);
      }
   },

   getInitialState: function () {
      return {
         notifications: [],
      };
   },

   propTypes: {
      style: PropTypes.oneOfType([PropTypes.bool, PropTypes.object]),
      noAnimation: PropTypes.bool,
      allowHTML: PropTypes.bool,
   },

   getDefaultProps: function () {
      return {
         style: {},
         noAnimation: false,
         allowHTML: false,
      };
   },

   addNotification: function (notification) {
      var _notification = Object.assign({}, Constants.notification, notification);
      var notifications = this.state.notifications;
      var i;

      if (!_notification.level) {
         throw new Error('notification level is required.');
      }

      if (Object.keys(Constants.levels).indexOf(_notification.level) === -1) {
         throw new Error("'" + _notification.level + "' is not a valid level.");
      }

      // eslint-disable-next-line
      if (isNaN(_notification.autoDismiss)) {
         throw new Error("'autoDismiss' must be a number.");
      }

      if (Object.keys(Constants.positions).indexOf(_notification.position) === -1) {
         throw new Error("'" + _notification.position + "' is not a valid position.");
      }

      // Some preparations
      _notification.position = _notification.position.toLowerCase();
      _notification.level = _notification.level.toLowerCase();
      _notification.autoDismiss = parseInt(_notification.autoDismiss, 10);

      _notification.uid = _notification.uid || this.uid;
      _notification.ref = 'notification-' + _notification.uid;
      this.uid += 1;

      // do not add if the notification already exists based on supplied uid
      for (i = 0; i < notifications.length; i += 1) {
         if (notifications[i].uid === _notification.uid) {
            return false;
         }
      }

      notifications.push(_notification);

      if (typeof _notification.onAdd === 'function') {
         notification.onAdd(_notification);
      }

      this.setState({
         notifications: notifications,
      });

      return _notification;
   },

   getNotificationRef: function (notification) {
      var self = this;
      var foundNotification = null;

      Object.keys(this.refs).forEach(function (container) {
         if (container.indexOf('container') > -1) {
            Object.keys(self.refs[container].refs).forEach(function (_notification) {
               var uid = notification.uid ? notification.uid : notification;
               if (_notification === 'notification-' + uid) {
                  // NOTE: Stop iterating further and return the found notification.
                  // Since UIDs are uniques and there won't be another notification found.
                  foundNotification = self.refs[container].refs[_notification];
               }
            });
         }
      });

      return foundNotification;
   },

   removeNotification: function (notification) {
      var foundNotification = this.getNotificationRef(notification);
      return foundNotification && foundNotification._hideNotification();
   },

   editNotification: function (notification, newNotification) {
      var foundNotification = null;
      var foundNotificationIndex = -1;
      // NOTE: Find state notification to update by using
      // `setState` and forcing React to re-render the component.
      var uid = notification.uid ? notification.uid : notification;

      var newNotifications = this.state.notifications.filter(function (stateNotification, index) {
         if (uid === stateNotification.uid) {
            foundNotification = stateNotification;
            foundNotificationIndex = index;
            return false;
         }

         return true;
      });

      if (!foundNotification) {
         return;
      }

      newNotifications.splice(foundNotificationIndex, 1, Object.assign({}, foundNotification, newNotification));

      this.setState({
         notifications: newNotifications,
      });
   },

   clearNotifications: function () {
      var self = this;
      Object.keys(this.refs).forEach(function (container) {
         if (container.indexOf('container') > -1) {
            Object.keys(self.refs[container].refs).forEach(function (_notification) {
               self.refs[container].refs[_notification]._hideNotification();
            });
         }
      });
   },

   componentDidMount: function () {
      this._getStyles.setOverrideStyle(this.props.style);
      this._isMounted = true;
   },

   componentWillUnmount: function () {
      this._isMounted = false;
   },

   render: function () {
      var self = this;
      var containers = null;
      var notifications = this.state.notifications;

      if (notifications.length) {
         containers = Object.keys(Constants.positions).map(function (position) {
            var _notifications = notifications.filter(function (notification) {
               return position === notification.position;
            });

            if (!_notifications.length) {
               return null;
            }

            return (
               <NotificationContainer
                  ref={'container-' + position}
                  key={position}
                  position={position}
                  notifications={_notifications}
                  getStyles={self._getStyles}
                  onRemove={self._didNotificationRemoved}
                  noAnimation={self.props.noAnimation}
                  allowHTML={self.props.allowHTML}
               />
            );
         });
      }

      return (
         <div className='notifications-wrapper' style={this._getStyles.wrapper()}>
            {containers}
         </div>
      );
   },
});

export default NotificationSystem;
