/* global Ya */
import Stats from '@yandex-infracloud-ui/yasm-browser-agent';
import axios, { CancelToken, isCancel } from 'axios';
import actions from './api-actions';

export class API {
   static CancelToken = CancelToken;

   static isCancel = isCancel;

   constructor(config = {}) {
      this.axios = axios.create(config);
   }

   /**
    *
    * @param {object} request
    * @param {string} request.service
    * @param {string} request.action
    * @param {object} [request.params]
    * @param {object} [request.data]
    * @param {object} [request.query]
    * @param {object} [request.headers]
    * @param {number} [request.timeout]
    * @param {boolean} [cancelToken=false]
    * @returns {*}
    */
   request({ service, action, params, data, query, headers, timeout }, cancelToken = false) {
      const { services } = window.CONFIG;

      let serviceName = service;
      // временное решение до переезда api
      if (service === 'yp' && params) {
         const location = params?.location ?? 'xdc';
         if (location !== 'xdc') {
            serviceName = `yp_${location}`;
         }
      }

      const { endpoint } = services.find(item => item.name === serviceName);
      const actionConfig = actions[serviceName][action];
      const { method } = actionConfig;
      let { path } = actionConfig;

      if (typeof path === 'function') {
         path = path(params);
      }

      if (actionConfig.headers) {
         let configHeaders = actionConfig.headers;

         if (typeof configHeaders === 'function') {
            configHeaders = configHeaders();
         }

         headers = {
            ...configHeaders,
            ...headers,
         };
      }

      const url = endpoint + path;

      const request = this.axios({
         method,
         url,
         data,
         params: query,
         headers,
         cancelToken,
         withCredentials: actionConfig.credentials,
         timeout: timeout ?? 60 * 1000,
      });

      return request
         .then(response => response.data)
         .catch(error => {
            if (actionConfig.onError) {
               actionConfig.onError(error, request);
            }

            if (this.isCancel(error)) {
               console.info('[api]: request canceled');
            }

            if (error?.response?.status >= 500 || error?.response?.status === undefined) {
               let additional = undefined;
               Stats.incrementCounter(`errors.network.${service}`, 'mmmm');
               Stats.incrementCounter(`errors.network.${service}.${action}`, 'mmmm');

               if (error?.response?.status >= 500) {
                  Stats.incrementCounter(`errors.network.${service}.5xx`, 'mmmm');
                  Stats.incrementCounter(`errors.network.${service}.${action}.5xx`, 'mmmm');

                  additional = {
                     status: error.response.status,
                     data,
                     query,
                  };
               }

               if (service.startsWith('yp')) {
                  const requestId = headers?.['X-YT-Request-Id'];
                  additional = {
                     ...additional,
                     requestId,
                  };
               }

               Ya.Rum.logError({
                  message: `API Error: ${service}`,
                  type: 'network',
                  level: Ya.Rum.ERROR_LEVEL.ERROR,
                  source: service,
                  sourceMethod: action,
                  additional,
               });
            }

            throw error;
         });
   }

   createCancelSource() {
      return API.CancelToken.source();
   }

   createCancelToken(...args) {
      return new API.CancelToken(...args);
   }

   isCancel(...args) {
      return API.isCancel(...args);
   }
}

export default new API();
