import { sortHandler } from '@yandex-infracloud-ui/libs';
import { isCancel } from 'axios';
import { get, uniq } from 'lodash';
import api from '../../services/api';
import { createApiActionTypes } from '../utils/api';
import { fetchAbcService } from './abc';
import { fetchGroups } from './staff';

let balancersFetchTimer;
const NAMESPACE_FETCH_TIMEOUT = 5000;
const cancelationSources = [];

/* --- Constants ---*/
export const domains = ['.yandex.net', '.yandex-team.ru', '.in.yandex-team.ru', '.yandex.ru'];
export const DNSBalancerZones = ['in.yandex-team.ru', 'in.yandex.net'];
export const capacityConfigurations = [
   {
      key: 'nano',
      val: 'nano',
      text:
         'NANO (requires 0.5 CPU cores, 1.5 Gb RAM, 10 Gb HDD for logs + 5.2 Gb HDD for runtime, 15 Mb/s HDD bandwidth guarantee)',
   },
   {
      key: 'micro',
      val: 'micro',
      text:
         'MICRO (requires 1 CPU core, 2 Gb RAM, 10 Gb HDD for logs + 5.2 Gb HDD for runtime, 15 Mb/s HDD bandwidth guarantee)',
   },
   {
      key: 'small',
      val: 'small',
      text:
         'SMALL (requires 2 CPU cores, 4 Gb RAM, 20 Gb HDD for logs + 5.2 Gb HDD for runtime, 20 Mb/s HDD bandwidth guarantee)',
   },
   {
      key: 'medium',
      val: 'medium',
      text:
         'MEDIUM (requires 4 CPU cores, 8 Gb RAM, 40 Gb HDD for logs + 5.2 Gb HDD for runtime, 30 Mb/s HDD bandwidth guarantee)',
   },
];
export const balancerMode = {
   quickStart: 'QUICK_START_MODE',
   other: 'OTHER',
};

/* --- Formatters for balancer union data --- */
const formatRewrite = ({ path }) => {
   const { enabled, src, dest } = path;

   return enabled ? { path: { src, dest } } : undefined;
};

const formatEndpointSets = ({ cluster, name }) => ({ cluster, id: name });

const formatRoute = ({ matcher, rewrite, outerBalancingOptions, innerBalancingOptions, id, endpointSets }) => ({
   matcher,
   rewrite: formatRewrite(rewrite),
   outerBalancingOptions,
   innerBalancingOptions,
   id,
   endpointSets: endpointSets.map(formatEndpointSets),
});

const formatDomain = ({ cert, id, matcher, routes }) => ({
   cert,
   id,
   matcher,
   routes: routes.map(formatRoute),
});

/* --- Reducer actions --- */
const LIST_NAMESPACES = createApiActionTypes('balancer', 'LIST_NAMESPACES');
const LIST_ALL_NAMESPACES = createApiActionTypes('balancer', 'LIST_ALL_NAMESPACES');
const LIST_BALANCER_PANELS = createApiActionTypes('balancer', 'LIST_BALANCER_PANELS');
const GET_QUICK_START_BALANCER_UNION = createApiActionTypes('balancer', 'GET_QUICK_START_BALANCER_UNION');
const RESET_STAGE_NAMESPACES = 'balancer/RESET_STAGE_NAMESPACES';
const GET_NAMESPACE = createApiActionTypes('balancer', 'GET_NAMESPACE');
const GET_BALANCERS = createApiActionTypes('balancer', 'GET_BALANCERS');
const GET_L3_BALANCERS = createApiActionTypes('balancer', 'GET_L3_BALANCERS');
const GET_BACKENDS = createApiActionTypes('balancer', 'GET_BACKENDS');
const GET_UPSTREAMS = createApiActionTypes('balancer', 'GET_UPSTREAMS');
const GET_DNS_RECORDS = createApiActionTypes('balancer', 'GET_DNS_RECORDS');
const GET_CERTIFICATES = createApiActionTypes('balancer', 'GET_CERTIFICATES');
const RESET_NAMESPACE = createApiActionTypes('balancer', 'RESET_NAMESPACE');
const START_LOADING = createApiActionTypes('balancer', 'START_LOADING');

/* --- Actions --- */
export const stopBalancersFetch = () => dispatch => {
   clearTimeout(balancersFetchTimer);
   cancelationSources.forEach(cancelSource => {
      cancelSource.cancel();
   });
   dispatch({ type: RESET_NAMESPACE.SUCCESS });
};

export const getNamespacesForEndpointSets = async endpointSets => {
   const quickStartNamespaces = await api
      .request({
         service: 'awacs',
         action: 'ListBalancers',
         data: {
            query: {
               ypEndpointSetFullIdIn: endpointSets,
            },
         },
      })
      .then(({ balancers: balancers_ }) => balancers_)
      .then(balancers_ => balancers_.map(({ meta: { namespaceId } }) => namespaceId))
      .then(namespaceIds => uniq(namespaceIds))
      .then(namespaceIds =>
         namespaceIds.map(namespaceId => ({
            mode: balancerMode.quickStart,
            namespaceId,
         })),
      );

   const nonQuickStartNamespaces = await api
      .request({
         service: 'awacs',
         action: 'BalancersNamespacesListNonQuickStart',
         data: {
            query: {
               ypEndpointSetFullIdIn: endpointSets,
            },
         },
      })
      .then(({ backends }) => backends)
      .then(backends => backends.map(({ meta: { namespaceId } }) => namespaceId))
      .then(result => uniq(result))
      .then(namespaceIds =>
         namespaceIds
            .map(namespaceId => ({
               mode: balancerMode.other,
               namespaceId,
            }))
            .filter(
               ({ namespaceId }) => !quickStartNamespaces.find(qsNamespace => qsNamespace.namespaceId === namespaceId),
            ),
      );

   return quickStartNamespaces.concat(nonQuickStartNamespaces);
};

export const fetchNamespaces = async (dispatch, endpointSets) => {
   const namespaces = await getNamespacesForEndpointSets(endpointSets);

   dispatch({ type: LIST_NAMESPACES.SUCCESS, data: namespaces });
};

const getStaffId = async abcServiceId => {
   // noinspection UnnecessaryLocalVariableJS
   const result = await api
      .request({
         service: 'staff',
         action: 'Groups',
         query: {
            'service.id': abcServiceId,
            '_one': 1,
            '_fields': 'id',
         },
      })
      .then(({ id }) => id)
      .catch(() => ({
         isSuccess: false,
         title: 'Staff API error',
         message: 'Error while fetching ABC service id using Staff API.',
      }));

   return result;
};

export const createBalancer = async (
   stageId,
   login,
   serviceId,
   alertRecipientsGroupId,
   fqdnData,
   endpointSetsData,
   balancerInstancesData,
) => {
   const staffId = await getStaffId(serviceId);

   const requestData = {
      meta: {
         abcServiceId: serviceId,
         auth: {
            staff: {
               owners: {
                  groupIds: [staffId.toString()],
                  logins: [login],
               },
            },
            type: 'STAFF',
         },
         category: `yandexdeploy/${stageId.replace(new RegExp('-', 'g'), '_').toLowerCase()}`,
         id: fqdnData.id,
         itsKnobsSyncEnabled: false,
      },
      order: {
         certificateOrderContent: {
            abcServiceId: serviceId,
            caName: 'InternalCA',
            commonName: fqdnData.commonName,
         },
         endpointSets: endpointSetsData,
         alertingSimpleSettings: {
            notifyStaffGroupId: alertRecipientsGroupId,
         },
         flowType: 'QUICK_START',
         ypLiteAllocationRequest: balancerInstancesData,
         doNotCreateUserEndpointSets: true,
      },
   };

   if (fqdnData.dnsRecordRequest) {
      requestData.order.dnsRecordRequest = fqdnData.dnsRecordRequest;
      requestData.order.certificateOrderContent.subjectAlternativeNames = fqdnData.subjectAlternativeNames;
   }

   // noinspection UnnecessaryLocalVariableJS
   const result = await api
      .request({
         service: 'awacs',
         action: 'CreateBalancer',
         data: requestData,
      })
      .then(({ namespace: { order: { status: { status } } } }) => ({ isSuccess: status === 'CREATED' }))
      .catch(({ response: { data = {} } = {}, message }) => ({
         isSuccess: false,
         title: message,
         message: data.message,
      }));

   return result;
};

/**
 * Обновляет спецификацию BalancerUnion'а в аваксе
 *
 * @param {*} meta
 * @param {*} domains_
 * @param {*} version
 * @param {*} http
 * @param {*} https
 * @returns
 */
export const updateBalancer = async (meta, domains_, version, http, https) => {
   const domainsFormatted = domains_.map(formatDomain);
   const quickStartBalancerMacro = {
      domains: domainsFormatted,
      version,
      http,
      https,
   };
   const requestData = {
      meta: { ...meta, comment: { value: 'Updated from Deploy UI' } },
      spec: {
         config: {
            value: {
               quickStartBalancerMacro,
            },
         },
      },
   };

   const result = {};
   await api
      .request({
         service: 'awacs',
         action: 'UpdateQuickStartBalancersUnion',
         data: requestData,
      })
      .then(() => {
         result.isSuccess = true;
      })
      .catch(({ response: { data = {} } = {}, message }) => {
         result.isSuccess = false;
         result.title = message;
         result.message = data.message;
      });

   return result;
};

export const getYASMPanels = async (dispatch, namespaceId, balancerId) => {
   const { aspectsSets } = await api.request({
      service: 'awacs',
      action: 'GetBalancersAspectSets',
      data: {
         id: balancerId,
         namespaceId,
      },
   });

   const panels = [];
   aspectsSets.forEach(
      ({
         content: {
            ui: { content },
         },
      }) => {
         if (content.yasmPanelUrl) {
            panels.push({
               title: 'Overview',
               url: content.yasmPanelUrl,
               id: 'overview',
            });
         }

         content.yasmPanels.forEach(({ desc, url, id }) => {
            panels.push({
               title: desc,
               url,
               id,
            });
         });
      },
   );

   dispatch({ type: LIST_BALANCER_PANELS.SUCCESS, panels, namespaceId, balancerId });
};

export const resetStageNamespaces = dispatch => {
   dispatch({ type: RESET_STAGE_NAMESPACES });
};

export const getQuickStartBalancersUnion = async (dispatch, namespaceId) => {
   const balancerUnion = await api.request({
      service: 'awacs',
      action: 'GetQuickStartBalancersUnion',
      query: {
         id: namespaceId,
      },
   });

   dispatch({ type: GET_QUICK_START_BALANCER_UNION.SUCCESS, balancerUnion });
};

export const fetchAllNamespaces = () => dispatch => {
   api.request({
      service: 'awacs',
      action: 'ListNamespaceSummaries',
      data: {
         fieldMask: 'meta.id,meta.category,meta.auth',
      },
   })
      .then(response => response.summaries)
      .then(summaries => summaries.sort((a, b) => sortHandler(a.category, b.category)))
      // Хак, выхлоп ListNamespaceSummaries привожу к формату ListNamespaces
      //  Просто не хочу переделывать legacy-код
      .then(summaries => summaries.map(s => ({ meta: s })))
      .then(namespaces => {
         dispatch({
            type: LIST_ALL_NAMESPACES.SUCCESS,
            data: namespaces,
         });
      });
};

export const fetchNamespace = namespaceId => dispatch => {
   dispatch({ type: START_LOADING.SUCCESS, data: 'namespace' });

   const cancelSource = api.createCancelSource();
   const cancelationToken = cancelSource.token;

   cancelationSources.push(cancelSource);

   api.request(
      {
         service: 'awacs',
         action: 'GetNamespace',
         data: {
            consistency: 'STRONG',
            id: namespaceId,
         },
      },
      cancelationToken,
   )
      .then(response => response.namespace)
      .catch(e => {
         if (!isCancel(e)) {
            if (e.response.status === 404) {
               return;
            }
            throw e;
         }
      })
      .then(namespace => {
         dispatch({
            type: GET_NAMESPACE.SUCCESS,
            data: namespace,
         });

         if (!namespace) {
            return;
         }

         const flowType = get(namespace, 'order.content.flowType');

         const status = get(namespace, 'order.status.status');
         const isReady = !namespace.order || ['FINISHED', 'CANCELLED'].includes(status);

         if (balancersFetchTimer) {
            clearTimeout(balancersFetchTimer);
         }

         if (isReady) {
            dispatch(fetchBalancers(namespaceId, cancelationToken));
            dispatch(fetchL3Balancers(namespaceId, cancelationToken));
            dispatch(fetchBackends(namespaceId, cancelationToken));
            if (flowType !== 'QUICK_START') {
               dispatch(fetchUpstreams(namespaceId, cancelationToken));
            }
            dispatch(fetchDNSRecords(namespaceId, cancelationToken));
            dispatch(fetchCertificates(namespaceId, cancelationToken));

            dispatch(fetchAbcService({ params: { serviceId: namespace.meta.abcServiceId } }));

            const {
               meta: {
                  auth: { staff: { owners: { groupIds = [] } = {} } = {} },
               },
            } = namespace;
            if (groupIds.length) {
               dispatch(fetchGroups({ params: { groupIds } }));
            }
         } else {
            balancersFetchTimer = setTimeout(() => {
               dispatch(fetchNamespace(namespaceId));
            }, NAMESPACE_FETCH_TIMEOUT);
         }
      });
};

export const fetchDNSRecords = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'ListDnsRecords',
         data: {
            namespaceId,
         },
      },
      cancelationToken,
   )
      .then(response => response.dnsRecords)
      .then(records => {
         dispatch({
            type: GET_DNS_RECORDS.SUCCESS,
            data: records,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const fetchUpstreams = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'ListUpstreams',
         data: {
            namespaceId,
            sortOrder: 'ASCEND',
            sortTarget: 'ORDER_LABEL',
         },
      },
      cancelationToken,
   )
      .then(response => response.upstreams)
      .then(upstreams => {
         dispatch({
            type: GET_UPSTREAMS.SUCCESS,
            data: upstreams,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const fetchBackends = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'ListBackends',
         data: {
            namespaceId,
            annotateWithEndpointSets: true,
         },
      },
      cancelationToken,
   )
      .then(response => response.annotatedBackends)
      .then(backends => {
         dispatch({
            type: GET_BACKENDS.SUCCESS,
            data: backends,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const fetchL3Balancers = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'GetL3Balancers',
         data: { namespaceId },
      },
      cancelationToken,
   )
      .then(response => response.l3Balancers)
      .then(balancers_ => {
         dispatch({
            type: GET_L3_BALANCERS.SUCCESS,
            data: balancers_,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const fetchBalancers = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'ListBalancers',
         data: {
            namespaceId,
         },
      },
      cancelationToken,
   )
      .then(response => response.balancers)
      .then(balancers_ => {
         dispatch({
            type: GET_BALANCERS.SUCCESS,
            data: balancers_,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const fetchCertificates = (namespaceId, cancelationToken) => dispatch => {
   api.request(
      {
         service: 'awacs',
         action: 'ListCertificates',
         data: {
            namespaceId,
         },
      },
      cancelationToken,
   )
      .then(response => response.certificates)
      .then(certs =>
         api
            .request({
               service: 'awacs',
               action: 'ListCertificateRenewals',
               data: {
                  namespaceId,
               },
            })
            .then(response => response.certificateRenewals)
            .then(renewalsList =>
               certs.reduce((result, currentCert) => {
                  const renewal = renewalsList.find(ren => ren.meta.id === currentCert.meta.id);

                  if (renewal) {
                     currentCert.renewal = renewal;
                  }

                  result.push(currentCert);

                  return result;
               }, []),
            ),
      )
      .then(certificates => {
         dispatch({
            type: GET_CERTIFICATES.SUCCESS,
            data: certificates,
         });
      })
      .catch(e => {
         if (!isCancel(e)) {
            throw e;
         }
      });
};

export const getUnifiedStatus = status => {
   const { active, inProgress, validated } = status;

   if (active.status === 'False' && inProgress.status === 'False' && validated.status === 'False') {
      return {
         status: 'error',
         text: 'Error',
      };
   }

   if (validated.status === 'True' && inProgress.status === 'True' && active.status === 'False') {
      return {
         status: 'progress',
         text: 'In progress',
      };
   }

   if (validated.status === 'True' && inProgress.status === 'False' && active.status === 'False') {
      return {
         status: 'progress',
         text: 'Validated',
      };
   }

   if (active.status === 'True' && inProgress.status === 'False' && validated.status === 'True') {
      return {
         status: 'ok',
         text: 'Active',
      };
   }

   if (validated.status === 'True' && inProgress.status === 'True' && active.status === 'True') {
      return {
         status: 'ok',
         text: 'Active',
      };
   }

   if (validated.status === 'Unknown' && inProgress.status === 'Unknown' && active.status === 'Unknown') {
      return {
         status: 'inactive',
         text: 'Not validated',
      };
   }

   return {
      status: 'unknown',
      text: 'Unknown',
   };
};

/* --- Reducer --- */
const initialState = {
   stageId: undefined,
   namespaces: undefined,
   stageNamespaces: undefined,
   namespace: undefined,
   balancerPanels: {},
   balancerUnion: undefined,
   updateTime: {
      namespace: undefined,
   },
   loadingFlags: {
      namespace: false,
   },
};

export default function balancers(state = initialState, action) {
   switch (action.type) {
      case LIST_NAMESPACES.SUCCESS: {
         return {
            ...state,
            stageNamespaces: action.data,
         };
      }
      case LIST_ALL_NAMESPACES.SUCCESS: {
         return {
            ...state,
            namespaces: action.data,
         };
      }
      case LIST_BALANCER_PANELS.SUCCESS: {
         const { panels, namespaceId, balancerId } = action;
         const key = `${namespaceId}:${balancerId}`;
         const newBalancerPanels = { ...state.balancerPanels, [key]: panels };
         return { ...state, balancerPanels: newBalancerPanels };
      }
      case RESET_STAGE_NAMESPACES: {
         return {
            ...state,
            stageNamespaces: undefined,
         };
      }
      case GET_QUICK_START_BALANCER_UNION.SUCCESS: {
         return {
            ...state,
            balancerUnion: action.balancerUnion,
         };
      }
      case GET_NAMESPACE.SUCCESS: {
         return {
            ...state,
            namespace: action.data,
            updateTime: {
               ...state.updated,
               namespace: Date.now(),
            },
            loadingFlags: {
               ...state.loadingFlags,
               namespace: false,
            },
         };
      }
      case GET_BALANCERS.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               balancers: action.data,
            },
         };
      }
      case GET_L3_BALANCERS.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               l3Balancers: action.data,
            },
         };
      }
      case GET_BACKENDS.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               backends: action.data,
            },
         };
      }
      case GET_UPSTREAMS.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               upstreams: action.data,
            },
         };
      }
      case GET_DNS_RECORDS.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               dnsRecords: action.data,
            },
         };
      }
      case GET_CERTIFICATES.SUCCESS: {
         return {
            ...state,
            namespace: {
               ...state.namespace,
               certificates: action.data,
            },
         };
      }
      case RESET_NAMESPACE.SUCCESS: {
         return {
            ...state,
            namespace: undefined,
            updateTime: {
               ...state.updateTime,
               namespace: undefined,
            },
         };
      }
      case START_LOADING.SUCCESS: {
         return {
            ...state,
            loadingFlags: {
               ...state.loadingFlags,
               [action.data]: true,
            },
         };
      }
      default: {
         return state;
      }
   }
}

export const selectNamespaces = (state, stageId) =>
   stageId === state.balancers.stageId ? state.balancers.stageNamespaces : [];

export const selectBalancerPanels = state => state.balancers.balancerPanels;
