import { TIMEOUTS } from '../../constants/api';
import api from '../../services/api';
import { createApiActionTypes } from '../utils/api';

/* --- ACTIONS --- */
const FETCH_SECRET = createApiActionTypes('deploy', 'FETCH_SECRET');
const FETCH_SECRET_VERSIONS = createApiActionTypes('deploy', 'FETCH_SECRET_VERSIONS');
const FETCH_DELEGATION_TOKEN = createApiActionTypes('deploy', 'FETCH_DELEGATION_TOKEN');
const FETCH_LOGS = createApiActionTypes('deploy', 'FETCH_LOGS');

/* --- ACTIONS --- */
export function fetchSecret({ params }) {
   const { secretId } = params;

   return function fetchSecretThunk(dispatch) {
      return api
         .request({
            service: 'deploy',
            action: 'Secrets',
            query: {
               query: secretId,
            },
         })
         .then(payload => {
            if (
               payload.secrets &&
               payload.secrets.length &&
               payload.secrets.filter(secret => secret.uuid === secretId).length
            ) {
               dispatch({
                  type: FETCH_SECRET.SUCCESS,
                  payload,
                  secretId,
               });
            }
         });
   };
}

export function fetchSecrets({ params }) {
   const { query } = params;

   return api.request({
      service: 'deploy',
      action: 'Secrets',
      query: {
         query,
      },
   });
}

export function fetchSecretVersions({ params }) {
   const { secretId } = params;

   return function fetchSecretVersionsThunk(dispatch) {
      return api
         .request({
            service: 'deploy',
            action: 'SecretVersions',
            query: {
               secretUuid: secretId,
            },
         })
         .then(payload => {
            dispatch({
               type: FETCH_SECRET_VERSIONS.SUCCESS,
               payload,
               secretId,
            });
         })
         .catch(error => {
            if (error.response.status === 403) {
               dispatch({
                  type: FETCH_SECRET_VERSIONS.FAILURE,
                  status: error.response.status,
                  secretId,
               });
            }
         });
   };
}

export function createDelegationToken({ params }) {
   const { secretId, signature } = params;

   return function createDelegationTokenThunk(dispatch) {
      return api
         .request({
            service: 'deploy',
            action: 'DelegationToken',
            query: {
               secretUuid: secretId,
               signature,
            },
         })
         .then(payload => {
            dispatch({
               type: FETCH_DELEGATION_TOKEN.SUCCESS,
               payload: payload.token,
               secretId,
               signature,
            });
         });
   };
}

export function fetchLogs({ params }) {
   const { filters, ...fetchParams } = params;
   const { stageId, deployUnitId, continuationToken } = params;

   return function fetchLogsThunk(dispatch) {
      return api
         .request({
            service: 'deploy',
            action: 'SearchLogEntries',
            data: {
               ...fetchParams,
            },
            query: {
               '_snake_case': 'true',
            },
            timeout: TIMEOUTS.LOGS,
         })
         .then(payload => {
            dispatch({
               type: FETCH_LOGS.SUCCESS,
               payload,
               stageId,
               deployUnitId,
               filters,
               ...(continuationToken ? { continuationToken } : {}),
            });
         })
         .catch(error => {
            const errorDetails = {};

            if (error.response) {
               if (error.response.data && error.response.data.message) {
                  errorDetails.message = error.response.data.message;
               } else if (error.response.statusText) {
                  errorDetails.message = error.response.statusText;
               }
            }

            dispatch({
               type: FETCH_LOGS.FAILURE,
               stageId,
               deployUnitId,
               filters,
               ...(continuationToken ? { continuationToken } : {}),
               error: errorDetails,
            });
         });
   };
}

/* --- REDUCER --- */
const initialState = {
   secrets: {},
   secretVersions: {},
   delegationTokens: {},
   logs: {},
};

export default function deploy(state = initialState, action) {
   switch (action.type) {
      case FETCH_SECRET.SUCCESS: {
         return {
            ...state,
            secrets: {
               ...state.secrets,
               [action.secretId]: action.payload.secrets.filter(secret => secret.uuid === action.secretId)[0] || {
                  uuid: action.secretId,
               },
            },
         };
      }
      case FETCH_SECRET_VERSIONS.FAILURE: {
         return {
            ...state,
            secrets: {
               ...state.secrets,
               [action.secretId]: {
                  uuid: action.secretId,
                  status: action.status,
                  forbidden: action.status === 403, // TODO: перейти со status на forbidden
               },
            },
         };
      }
      case FETCH_SECRET_VERSIONS.SUCCESS: {
         return {
            ...state,
            secretVersions: {
               ...state.secretVersions,
               [action.secretId]: action.payload.versions,
            },
         };
      }
      case FETCH_DELEGATION_TOKEN.SUCCESS: {
         const { delegationTokens } = state;
         return {
            ...state,
            delegationTokens: {
               ...delegationTokens,
               [action.signature]: {
                  ...delegationTokens[action.signature],
                  [action.secretId]: action.payload.token,
               },
            },
         };
      }
      case FETCH_LOGS.SUCCESS: {
         const { logs } = state;
         const continuationTokens = action.payload.continuation_tokens || action.payload.continuationTokens || {};

         let logEntries = (action.payload.log_entries || action.payload.logEntries).map(log => ({
            log,
            linkToken: continuationTokens.current,
            logMessage: log.message || log.log_message || log.logMessage || '', // TODO: DEPLOY-2318 AWAIT: DEPLOY-2589
            logLevel: log.log_level || log.logLevel || '', // убрать лишние поля, когда snake_case выедет в продакшен
            loggerName: log.logger_name || log.loggerName || '',
            timestamp: log.timestamp,
            box: log.box,
            workload: log.workload,
         }));

         if (action.continuationToken) {
            const stateLogs =
               logs && logs[action.stageId] && logs[action.stageId][action.deployUnitId]
                  ? logs[action.stageId][action.deployUnitId]
                  : {};
            const stateContinuationTokens = stateLogs.continuationTokens;
            const stateLogEntries = stateLogs.logEntries;

            if (
               stateContinuationTokens &&
               stateContinuationTokens.forward &&
               stateContinuationTokens.forward === action.continuationToken
            ) {
               if (stateContinuationTokens.backward) {
                  continuationTokens.backward = stateContinuationTokens.backward;
                  logEntries = stateLogEntries.concat(logEntries);
               }
            }
         }

         return {
            ...state,
            logs: {
               ...logs,
               [action.stageId]: {
                  [action.deployUnitId]: {
                     logEntries,
                     continuationTokens,
                     filters: action.filters,
                  },
               },
            },
         };
      }
      case FETCH_LOGS.FAILURE: {
         const { logs } = state;
         let logEntries = [];

         if (action.continuationToken) {
            const stateLogs =
               logs && logs[action.stageId] && logs[action.stageId][action.deployUnitId]
                  ? logs[action.stageId][action.deployUnitId]
                  : {};

            if (stateLogs.logEntries) {
               logEntries = stateLogs.logEntries;
            }
         }

         return {
            ...state,
            logs: {
               ...logs,
               [action.stageId]: {
                  [action.deployUnitId]: {
                     logEntries,
                     filters: action.filters,
                     error: action.error,
                  },
               },
            },
         };
      }
      default:
         return state;
   }
}

export const selectSecrets = state => state.deploy.secrets;

export const selectSecretVersions = state => state.deploy.secretVersions;

export const selectDelegationTokens = state => state.deploy.delegationTokens;

export const selectLogs = (state, stageId) => state.deploy.logs[stageId];
