import { cloneDeep } from 'lodash';
import api from '../../services/api';
import { createApiActionTypes } from '../utils/api';

/* --- ACTIONS --- */
const FETCH_ITYPES = createApiActionTypes('yasm', 'FETCH_ITYPES');
const FETCH_TAG_KEYS = createApiActionTypes('yasm', 'FETCH_TAG_KEYS');
const FETCH_TAG_VALUES = createApiActionTypes('yasm', 'FETCH_TAG_VALUES');

export const DEPLOY_DEFINED_TAGS = { 'stage': true, 'deploy_unit': true, 'box': true, 'volume': true };

export const fetchITypes = (itype, selectedTagNames) => dispatch =>
   api
      .request({
         service: 'yasm',
         action: 'IType',
      })
      .then(({ response }) => response.result)
      .then(iTypes => {
         dispatch({
            type: FETCH_ITYPES.SUCCESS,
            iTypes,
         });

         if (itype) {
            dispatch(fetchTagNames(itype, selectedTagNames));
         }
      });

export const fetchTagNames = (itype, selectedTagNames = []) => dispatch =>
   api
      .request({
         service: 'yasm',
         action: 'TagKeys',
         query: {
            itype,
         },
      })
      .then(itypeTagsData =>
         itypeTagsData[itype].reduce((result, tagName) => {
            if (!DEPLOY_DEFINED_TAGS[tagName]) {
               result[tagName] = [];
            }
            return result;
         }, {}),
      )
      .then(tags => {
         dispatch({
            type: FETCH_TAG_KEYS.SUCCESS,
            itype,
            tags,
         });

         if (selectedTagNames.length) {
            selectedTagNames.forEach(tagName => {
               dispatch(fetchTagValues(itype, tagName));
            });
         }
      });

export const fetchTagValues = (itype, tagName) => (dispatch, getState) => {
   const {
      yasm: { tags },
   } = getState();
   const itypeTagsObject = tags[itype] || {};
   const existingTagValues = itypeTagsObject[tagName] || [];

   if (existingTagValues.length > 0) {
      return Promise.resolve();
   }

   return api
      .request({
         service: 'yasm',
         action: 'Tags',
         query: {
            itype,
            fields: tagName,
         },
      })
      .then(({ response: { result }, status }) => {
         if (status === 'ok') {
            const tagValues = result.map(x => x[tagName]).filter(x => x);
            dispatch({
               type: FETCH_TAG_VALUES.SUCCESS,
               itype,
               tagName,
               tagValues,
            });
         }
      });
};

/* --- REDUCER --- */
const initialState = {
   iTypes: [],
   tags: {},
};

const updateTagValues = ({ itype, tagName, tagValues }, state) => {
   const itypeTags = cloneDeep(state.tags[itype]);
   itypeTags[tagName] = tagValues;

   return {
      ...state,
      tags: {
         ...state.tags,
         [itype]: itypeTags,
      },
   };
};

const updateTags = ({ itype, tags }, state) => {
   const newTags = cloneDeep(state.tags);
   newTags[itype] = tags;

   return { ...state, tags: newTags };
};

export default function yasm(state = initialState, action) {
   switch (action.type) {
      case FETCH_ITYPES.SUCCESS:
         return {
            ...state,
            iTypes: action.iTypes,
         };
      case FETCH_TAG_KEYS.SUCCESS:
         return updateTags(action, state);
      case FETCH_TAG_VALUES.SUCCESS:
         return updateTagValues(action, state);
      default:
         return state;
   }
}

/* --- SELECTORS --- */
export const getITypes = state => state.yasm.iTypes;
export const getTags = state => state.yasm.tags;
