import { classNames, ExternalLink, parseApiError, useDismounted, useTitle } from '@yandex-infracloud-ui/libs';
import { noop } from 'lodash-es';
import React, { useCallback, useContext, useEffect, useState } from 'react';
import { RouteComponentProps } from 'react-router';
import { Observable } from 'rxjs';
import { finalize, takeUntil } from 'rxjs/operators';

import { LegoButton } from '../../../_lego';
import modalClasses from '../../../design/commonModals.module.css';
import { EXTERNAL_LINKS } from '../../../models';
import { preferredPatcherRevision } from '../../../models/ui';
import { NotificationsContext } from '../../../old-code/services/Notifications';
import { infraComponentsApi } from '../../../services';
import { RevisionsMap, SidecarType } from '../../../services/api/services/InfracomponentsApi';

import styles from './InfraComponents.module.css';

const APPLY_ON_PERCENT = [5, 10, 25, 50, 100];

export const InfraComponentsPage: React.FC<RouteComponentProps<{ sidecar: SidecarType }>> = React.memo(({ match }) => {
   useTitle(`Infra components`);

   const { sidecar } = match.params;
   const isPatcher = sidecar === SidecarType.Patchers;

   const [currentRevisions, setCurrentRevisions] = useState<RevisionsMap>({});
   const [isLoading, setIsLoading] = useState(false);
   const notifications = useContext(NotificationsContext);
   const dismounted = useDismounted();

   const handleOk = (percent: number) => {
      setIsLoading(true);
      let infraRequest: Observable<string>;
      if (isPatcher) {
         infraRequest = infraComponentsApi.applyPatcher(preferredPatcherRevision, percent);
      } else {
         infraRequest = infraComponentsApi.applySidecar(sidecar, percent);
      }

      infraRequest.pipe(finalize(() => setIsLoading(false))).subscribe(noop, resp => {
         notifications.add({
            level: resp.status === 200 ? 'success' : 'error',
            title: resp.status === 200 ? 'updated' : 'error',
            message: parseApiError(resp.message),
            autoDismiss: 0,
         });
      });
   };

   useEffect(() => {
      if (isPatcher) {
         infraComponentsApi
            .getStatPatchers()
            .pipe(takeUntil(dismounted))
            .subscribe(resp => {
               setCurrentRevisions(resp);
            });
      } else {
         infraComponentsApi
            .getStatSidecars(sidecar)
            .pipe(takeUntil(dismounted))
            .subscribe(resp => {
               setCurrentRevisions(resp);
            });
      }
   }, [dismounted, isPatcher, sidecar]);

   const renderRevision = useCallback(
      (revision: string) =>
         !isPatcher && /\d+/.test(revision) ? (
            <ExternalLink href={EXTERNAL_LINKS.sandboxResource(Number.parseInt(revision, 10))}>{revision}</ExternalLink>
         ) : (
            revision
         ),
      [isPatcher],
   );

   const renderItem = useCallback(
      (revision: string, values: Array<String>) => (
         <div>
            <div>
               {values.length} deploy unit(s) under {renderRevision(revision)} revision{' '}
            </div>
            <ol>{values ? values.map(name => <li>{name}</li>) : ''}</ol>
         </div>
      ),
      [renderRevision],
   );

   return (
      <div className={styles.page}>
         <p>
            {Object.keys(currentRevisions).length} different {sidecar} revisions is used
         </p>
         <p>
            {APPLY_ON_PERCENT.map(p => (
               <LegoButton
                  theme={'action'}
                  size={'s'}
                  view={'default'}
                  tone={'default'}
                  cls={classNames(modalClasses.button, modalClasses.danderButton)}
                  onClick={() => handleOk(p)}
                  disabled={isLoading}
               >
                  Apply for {p}% DU
               </LegoButton>
            ))}
         </p>

         <div> {Object.keys(currentRevisions).map(k => renderItem(k, currentRevisions[k] || []))}</div>
      </div>
   );
});

InfraComponentsPage.displayName = 'Infra components statistic';
