import { Spin } from '@yandex-cloud/uikit';
import { Collapse } from '@yandex-data-ui/common';
import { formatDate } from '@yandex-infracloud-ui/libs';
import React, { useEffect, useMemo, useState } from 'react';
import { useDispatch } from 'react-redux';

import { Page } from '../../components/layout/Page/Page';
import { fetchTicketsData } from '../../redux/slices/startrek/startrek';

import classes from './index.module.css';
import { TicketRow } from './TicketRow';

interface TicketData {
   [author: string]: {
      [date: string]: string[]; // times
   };
}

interface Changelog {
   [version: string]: {
      [ticketId: string]: string | TicketData;
   };
}

const versionRegex = /^\d+\.\d+\.\d+/; // simplified semver

const VersionList: React.FC<{ versions: string[]; ticketIds: string[]; changelog: Changelog }> = React.memo(
   ({ versions, ticketIds, changelog }) => {
      const dispatch = useDispatch();

      useEffect(() => {
         dispatch(fetchTicketsData({ ids: ticketIds }));
      }, [dispatch, ticketIds]);
      return (
         <>
            {versions.map(version => {
               const tickets = changelog[version];
               /* eslint-disable-next-line no-underscore-dangle */
               const date = tickets.__date as string | undefined;
               const ticketList = Object.keys(tickets)
                  .filter(e => !e.startsWith('__'))
                  .sort((a, b) => Number(a.split('-')[1]) - Number(b.split('-')[1]));

               return (
                  <div className={classes.version} key={version}>
                     <h2>
                        {version}
                        {date && <span className={classes.date}>{formatDate(new Date(date), 'dd.MM.yyyy HH:mm')}</span>}
                     </h2>
                     {ticketList.map(ticket => (
                        <TicketRow ticketId={ticket} ticketData={tickets[ticket] as TicketData} key={ticket} />
                     ))}
                  </div>
               );
            })}
         </>
      );
   },
);

VersionList.displayName = 'VersionList';

interface SemverValues {
   major: number;
   minor: number;
   patch: number;
}

function getVersionValues(version: string): SemverValues {
   const [major, minor, patch] = version.split('.').map(Number);
   return { major, minor, patch };
}

function getTicketIds(changelog: Changelog, versions: string[]) {
   return [...new Set(versions.flatMap(v => Object.keys(changelog[v])).filter(e => !e.startsWith('__')))];
}

export const ChangelogPage = React.memo(() => {
   const [changelog, setChangelog] = useState({} as Changelog);

   useEffect(() => {
      import('../../changelog.json').then(result => setChangelog(result.default));
   }, []);

   const versions = useMemo(
      () =>
         Object.keys(changelog)
            .filter(
               version =>
                  changelog[version] && Object.keys(changelog[version]).length > 0 && versionRegex.test(version),
            )
            .sort((a, b) => {
               const { major: aMajor, minor: aMinor, patch: aPatch } = getVersionValues(a);
               const { major: bMajor, minor: bMinor, patch: bPatch } = getVersionValues(b);
               return aMajor > bMajor || aMinor > bMinor || aPatch > bPatch ? -1 : 1;
            }),
      [changelog],
   );

   const { major: lastMajor, minor: lastMinor } = getVersionValues(versions[0] ?? '0.0.0');

   const newVersions = useMemo(
      () =>
         versions.filter(e => {
            const { major, minor } = getVersionValues(e);
            return major === lastMajor && minor === lastMinor;
         }),
      [lastMajor, lastMinor, versions],
   );

   const newVersionsSet = useMemo(() => new Set(newVersions), [newVersions]);

   const oldVersions = useMemo(() => versions.filter(e => !newVersionsSet.has(e)), [newVersionsSet, versions]);

   const newTicketIds = useMemo(() => getTicketIds(changelog, newVersions), [changelog, newVersions]);

   const newTicketIdsSet = useMemo(() => new Set(newTicketIds), [newTicketIds]);

   const oldTicketIds = useMemo(() => getTicketIds(changelog, oldVersions).filter(e => !newTicketIdsSet.has(e)), [
      changelog,
      newTicketIdsSet,
      oldVersions,
   ]);

   return (
      <Page title={'UI Changelog'} header={<h1>UI Changelog</h1>}>
         {versions.length > 0 ? (
            <>
               <VersionList versions={newVersions} ticketIds={newTicketIds} changelog={changelog} />
               <Collapse title={'Old versions'}>
                  <VersionList versions={oldVersions} ticketIds={oldTicketIds} changelog={changelog} />
               </Collapse>
            </>
         ) : (
            <Spin />
         )}
      </Page>
   );
});

ChangelogPage.displayName = 'ChangelogPage';
