import React, { useCallback, useContext, useMemo, useRef } from 'react';

import { faSpinner, faUndo } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { fromQuery, useDismounted, useObservableValue, useTitle } from '@yandex-infracloud-ui/libs';
import { useSelector } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { takeUntil } from 'rxjs/operators';
import { Button } from '@yandex-cloud/uikit';

import { EditSharedNode, SharedNodeProvider } from '../../components/lib';
import { StageHugeForm } from '../../components/stage-huge-form';
import { StageHugeFormMode, StageHugeFormShowPage } from '../../components/stage-huge-form/models';
import { handleApiError, NotificationsContext, urlBuilder } from '../../models';
import { emptyRawStage, getEmptyStage, ProjectItem, Stage } from '../../models/ui';
import { TStage } from '../../proto-typings';
import type { RootState } from '../../redux';
import { useConfig, ypApi } from '../../services';
import { StageBreadcrumbs } from '../stages/_stageId/components';

import { MigrationBanner } from './components/MigrationBanner';
import classes from './NewStagePage.module.css';
import { RenderArgs } from '../../components/stage-huge-form/StageHugeForm';

const renderChildren = (form: RenderArgs) => (
   <>
      <EditSharedNode replace={true}>
         {/* {form.contentEditViewSwitcher} */}

         {form.showPage === StageHugeFormShowPage.Edit && form.hasUnsavedChanges ? (
            <Button onClick={form.revert} extraProps={{ 'data-e2e': 'StageHugeForm:RevertUnsavedChangesButton' }}>
               <FontAwesomeIcon icon={faUndo} />
               &nbsp;&nbsp; Revert changes
            </Button>
         ) : null}

         <Button onClick={form.cancel} extraProps={{ 'data-e2e': 'StageHugeForm:CancelButton' }}>
            {form.showPage === StageHugeFormShowPage.Diff
               ? 'Close yaml'
               : form.showPage === StageHugeFormShowPage.Update
               ? 'Back'
               : 'Cancel'}
         </Button>

         <Button
            view={'action'}
            onClick={form.update}
            extraProps={{ 'data-e2e': 'StageHugeForm:UpdateButton' }}
            disabled={form.showPage === StageHugeFormShowPage.Diff || form.isSubmitting}
         >
            {form.isSubmitting ? (
               <>
                  <FontAwesomeIcon icon={faSpinner} spin={true} /> Checking
               </>
            ) : (
               'Review Changes'
            )}
         </Button>
      </EditSharedNode>

      {form.content}
   </>
);

interface QueryParams {
   projectId?: string;
}

export const NewStagePage: React.FC<RouteComponentProps> = React.memo(({ location, history, match }) => {
   useTitle('New stage');

   const { projectId } = fromQuery(location.search) as QueryParams;
   const projectIdRef = useRef(projectId); // ref for keeping projectId on next render
   const project = useProject(projectIdRef.current);

   const userLogin = useConfig()!.user.login;

   const rawStage = emptyRawStage as TStage;

   const stage: Stage = useMemo(
      () => ({
         ...getEmptyStage({ projectId: projectIdRef.current }),
         id: `${userLogin}-stage`,
         project,
      }),
      [userLogin, project],
   );

   const initialStageId = stage.id;
   const stageIdSelector = useCallback(
      (s: RootState) => {
         const record = s.formStates[match.url];

         return record ? (record.state.values.id as string) : initialStageId;
      },
      [initialStageId, match.url],
   );
   const newId = useSelector(stageIdSelector);

   const handleCancel = useCallback(() => {
      history.push(projectId ? urlBuilder.project(projectId) : urlBuilder.projects());
   }, [history, projectId]);

   const sharedNodeRef = useRef<HTMLDivElement>(null);

   // #DEPLOY-5640
   if (projectId && !project) {
      return null;
   }

   return (
      <>
         <MigrationBanner />

         <div className={classes.page}>
            <div className={classes.breadcrumbs}>
               <StageBreadcrumbs projectId={projectId} lastPart={newId ? `${newId} (new)` : 'New stage'} />

               <div className={classes.buttons} ref={sharedNodeRef} data-e2e={'StageHugeForm:Buttons'} />
            </div>

            <div className={classes.content}>
               <SharedNodeProvider nodeRef={sharedNodeRef}>
                  <StageHugeForm mode={StageHugeFormMode.New} rawStage={rawStage} stage={stage} onCancel={handleCancel}>
                     {renderChildren}
                  </StageHugeForm>
               </SharedNodeProvider>
            </div>
         </div>
      </>
   );
});

NewStagePage.displayName = 'NewStagePage';

function useProject(projectId: string | undefined): ProjectItem | null {
   const dismounted = useDismounted();
   const notifications = useContext(NotificationsContext);

   const actualProject = useObservableValue(
      ypApi.getProject(projectId).pipe(takeUntil(dismounted)),
      handleApiError('Load project', notifications) as any,
      [projectId],
   );

   // #DEPLOY-5640
   if (!projectId || !actualProject) {
      return null;
   }

   return actualProject;
}
