import { Loader } from '@yandex-cloud/uikit';
import React, { useMemo, useState } from 'react';
import { PageTemplate } from '../../components/yp/PageTemplate/PageTemplate';
import { YpAccountQuota } from '../../models/ui';
import PageWrapper from '../../old-code/components/PageWrapper/PageWrapper';
import { useNetworkErrors } from '../../redux';
import { useYpQuota } from '../../redux/hooks/useYpQuota';
import { getYPError } from '../yp/utils';
import { GraphNode, GraphNodeValue, GraphObject, OrderingKey, orderingOptions, QuotaOrder } from './models';
import { QuotaFilters, QuotaFiltersParams } from './QuotaFilters/QuotaFilters';

import classes from './Quotas.module.css';
import { Account } from './__account/Account';
import { Segment } from './__segment/Segment';
import { TreeNode } from './__treeNode/TreeNode';

function getTitle(quota: YpAccountQuota, key: OrderingKey) {
   switch (key) {
      case 'accountId':
         return <Account id={quota.accountId} name={quota.name} />;
      case 'cluster':
         return quota.cluster.toUpperCase();
      default:
         return <Segment segment={quota[key]} />;
   }
}

const createGraph = (groupingKeys: OrderingKey[], data: YpAccountQuota[]) => {
   const graph: GraphObject = {};

   data.forEach(item => {
      const [first, second, third] = groupingKeys;

      const firstValue = item[first];
      if (!graph[firstValue]) {
         const firstNode: GraphNode = {
            value: { [first]: firstValue, [second]: '', [third]: '' } as GraphNodeValue,
            children: {},
            title: getTitle(item, first),
            key: firstValue,
            level: 0,
         };

         graph[firstValue] = firstNode;
      }

      const secondValue = item[second];
      const firstChildren = graph[firstValue].children!;
      if (!firstChildren[secondValue]) {
         const secondNode: GraphNode = {
            value: { [first]: firstValue, [second]: secondValue, [third]: '' } as GraphNodeValue,
            children: {},
            title: getTitle(item, second),
            key: `${firstValue}:${secondValue}`,
            level: 1,
         };

         firstChildren[secondValue] = secondNode;
      }

      const thirdValue = item[third];
      const secondChildren = firstChildren[secondValue].children!;
      const thirdNode: GraphNode = {
         value: { [third]: thirdValue, [second]: secondValue, [first]: firstValue } as GraphNodeValue,
         title: getTitle(item, third),
         key: `${firstValue}:${secondValue}:${thirdValue}`,
         level: 2,
         quota: {
            limit: item.resourceLimits,
            usage: item.resourceUsage,
         },
      };

      secondChildren[thirdValue] = thirdNode;
   });

   return Object.values(graph);
};

export const Quotas: React.FC = () => {
   const { isAccountFetching, quotas, requestKeys } = useYpQuota();
   const [filtersState, setFilters] = useState<QuotaFiltersParams>({ order: QuotaOrder.Account });

   const filters = (
      <QuotaFilters
         filters={filtersState}
         onFiltersChange={newFilters => {
            setFilters(newFilters);
         }}
      />
   );

   const tree = useMemo(() => createGraph(orderingOptions[filtersState.order], Object.values(quotas)), [
      quotas,
      filtersState,
   ]);

   const errors = useNetworkErrors(requestKeys);
   const errorItems = useMemo(() => requestKeys.map(key => getYPError(errors[key])).filter(Boolean), [
      errors,
      requestKeys,
   ]);

   return (
      <div className={classes.container}>
         <PageWrapper title={`YP`}>
            <PageTemplate navigation={filters} error={errorItems.length ? errorItems : undefined}>
               <div className={classes.contentContainer}>
                  <div>
                     {tree.map(item => (
                        <TreeNode key={item.key} node={item} />
                     ))}
                  </div>
                  <div className={classes.loaderContainer}>{isAccountFetching && <Loader size={'s'} />}</div>
               </div>
            </PageTemplate>
         </PageWrapper>
      </div>
   );
};
