import { classNames } from '@yandex-infracloud-ui/libs';
import React, { useMemo } from 'react';
import { ResourceBar, ResourceBarProps } from '../../../components/yp';
import { TPerSegmentResourceTotals } from '../../../proto-typings';
import { formatBytes, formatCores } from '../../../utils/yp';
import classes from './AccountQuota.module.css';

interface Props {
   limit?: TPerSegmentResourceTotals;
   usage?: TPerSegmentResourceTotals;
   className?: string;
}

const getBarParams = (limit: TPerSegmentResourceTotals | undefined, usage: TPerSegmentResourceTotals | undefined) => {
   const result: (ResourceBarProps & { title: string })[] = [
      {
         total: (limit?.cpu?.capacity ?? 0) / 1000,
         used: (usage?.cpu?.capacity ?? 0) / 1000,
         title: 'CPU',
         suffix: ' vcores',
         format: formatCores,
      },
      {
         total: limit?.memory?.capacity ?? 0,
         used: usage?.memory?.capacity ?? 0,
         title: 'Memory',
         format: formatBytes,
      },
      {
         title: 'Network',
         total: limit?.network?.bandwidth ?? 0,
         used: usage?.network?.bandwidth ?? 0,
         format: formatBytes,
         suffix: '/s',
      },
      {
         title: 'IP addresses',
         total: limit?.internet_address?.capacity ?? 0,
         used: usage?.internet_address?.capacity ?? 0,
      },
   ];

   if (usage?.disk_per_storage_class?.hdd || limit?.disk_per_storage_class?.hdd) {
      const hddLimits = limit?.disk_per_storage_class.hdd;
      const hddUsage = usage?.disk_per_storage_class.hdd;

      result.push(
         {
            title: 'HDD capacity',
            total: hddLimits?.capacity ?? 0,
            used: hddUsage?.capacity ?? 0,
            format: formatBytes,
         },
         {
            title: 'HDD bandwidth',
            total: hddLimits?.bandwidth ?? 0,
            used: hddUsage?.bandwidth ?? 0,
            format: formatBytes,
            suffix: '/s',
         },
      );
   }

   if (usage?.disk_per_storage_class?.ssd || limit?.disk_per_storage_class?.ssd) {
      const ssdLimits = limit?.disk_per_storage_class.ssd;
      const ssdUsage = usage?.disk_per_storage_class.ssd;

      result.push(
         {
            title: 'SSD capacity',
            total: ssdLimits?.capacity ?? 0,
            used: ssdUsage?.capacity ?? 0,
            format: formatBytes,
         },
         {
            title: 'SSD bandwidth',
            total: ssdLimits?.bandwidth ?? 0,
            used: ssdUsage?.bandwidth ?? 0,
            format: formatBytes,
            suffix: '/s',
         },
      );
   }

   const gpuLimit = limit?.gpu_per_model;
   const gpuUsage = usage?.gpu_per_model ?? {};
   if (gpuLimit) {
      Object.keys(gpuLimit).forEach(model => {
         const { capacity } = gpuLimit[model];
         const modelUsage = gpuUsage[model];
         if (capacity !== 0) {
            result.push({
               title: `GPU (${model})`,
               total: capacity,
               used: modelUsage?.capacity ?? 0,
            });
         }
      });
   }

   return result;
};

export const AccountQuota: React.FC<Props> = ({ limit, usage, className }) => {
   const bars = useMemo(
      () =>
         getBarParams(limit, usage).map(props => (
            <div key={props.title} className={classes.barContainer}>
               <ResourceBar {...props} view={'thinnest'} />
            </div>
         )),
      [limit, usage],
   );

   return <div className={classNames(classes.quota, className)}>{bars}</div>;
};

AccountQuota.displayName = 'Quotas__AccountQuota';
