import { Card } from '@yandex-cloud/uikit';
import { Collapse, StaffCard } from '@yandex-data-ui/common';
import { EMPTY_VALUE, formatDate, isEmpty, UserName, UserOrGroupType } from '@yandex-infracloud-ui/libs';
import React, { useMemo } from 'react';

import { DefinitionList, DefinitionListItem } from '../../../../../components/lib';
import { DockerResources, ReleaseStatus, SandboxResources } from '../../../../../components/release';
import { Release } from '../../../../../models/ui';
import { parseSimpleYpTimestamp } from '../../../../../utils';

import classes from './ReleaseInfo.module.css';

interface Props {
   release: Release;
}

function getItems(release: Release): DefinitionListItem[] {
   const { title, description, creationTime, type } = release;

   return [
      {
         name: 'Title',
         content: !title || isEmpty(title) ? EMPTY_VALUE : title,
      },
      {
         name: 'Description',
         content: !description || isEmpty(description) ? EMPTY_VALUE : description,
      },
      {
         name: 'Created',
         content: <span>{formatDate(parseSimpleYpTimestamp(creationTime))}</span>,
      },
      {
         name: 'Release author',
         content: (
            <StaffCard login={release.author}>
               <UserName value={{ id: release.author, type: UserOrGroupType.People }} />
            </StaffCard>
         ),
      },
      {
         name: 'Type',
         content: type,
      },
      {
         name: 'Status',
         content: <ReleaseStatus status={release.status} isAnimated={true} />,
      },
   ];
}

export const ReleaseInfo: React.FC<Props> = React.memo(({ release }) => {
   const { sandbox, docker } = release;
   const items = useMemo(() => getItems(release), [release]);
   return (
      <div>
         <DefinitionList items={items} />
         <div className={classes.resources}>
            <Card view={'raised'} type={'container'} className={classes.resourcesCard}>
               <Collapse title={<h2>Resources</h2>} defaultIsExpand={false}>
                  <div className={classes.resourceTable}>
                     {sandbox && <SandboxResources sandbox={sandbox} />}
                     {docker && <DockerResources docker={docker} />}
                  </div>
               </Collapse>
            </Card>
         </div>
      </div>
   );
});

ReleaseInfo.displayName = 'ReleaseInfo';
