import React from 'react';
import { RouteComponentProps } from 'react-router';

import { EmptyContainer, EmptyContainerType, useTitle } from '@yandex-infracloud-ui/libs';
import { YpErrorTooltip } from '../../../components/network';
import { Page } from '../../../components/layout/Page/Page';
import LoadingPage from '../../../old-code/components/LoadingPage/LoadingPage';
import { useNetworkErrors, useRelease } from '../../../redux';
import { ReleaseInfo, ReleaseNotFound } from './components';
import { ReleaseDeployTickets } from './components/ReleaseDeployTickets/ReleaseDeployTickets';

import classes from './index.module.css';
import { ReleaseRequestKeys } from '../../../models/ui';
import { ReleasePromoCard } from '../../../components/release';

export const ReleaseIndexPage: React.FC<RouteComponentProps<{ releaseId: string }>> = React.memo(({ match }) => {
   const { releaseId } = match.params;
   const { release, meta } = useRelease(releaseId);

   useTitle(`Release ${releaseId}`);

   const requestKey = ReleaseRequestKeys.getOne({ id: releaseId ?? '' });
   const networkErrors = useNetworkErrors([requestKey]);
   const releaseError = networkErrors[requestKey];

   if (meta.isProcessing) {
      return <LoadingPage />;
   }

   if (!release) {
      if (meta.isNotFound) {
         return <ReleaseNotFound releaseId={releaseId} />;
      }
      if (releaseError) {
         return (
            <EmptyContainer
               type={EmptyContainerType.Error}
               description={<YpErrorTooltip error={releaseError.error!} request={releaseError.request} />}
            />
         );
      }
      return null;
   }

   return (
      <Page header={<h1>Release {releaseId}</h1>}>
         <div className={classes.content} data-e2e={'StageIndexPage:content'}>
            <div className={classes.info}>
               <ReleaseInfo release={release} />
               <div>
                  <ReleasePromoCard />
               </div>
            </div>
            <ReleaseDeployTickets releaseId={releaseId} />
         </div>
      </Page>
   );
});

ReleaseIndexPage.displayName = 'StageIndexPage';
