import React, { ReactNode, useCallback, useContext, useEffect, useRef, useState } from 'react';

import {
   CheckboxField2,
   classNames,
   DISMISS_REASON,
   IModalProps,
   Loader,
   ModalLayout,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';
import { Form, Formik } from 'formik';
import { FormikHelpers } from 'formik/dist/types';
import { useDispatch } from 'react-redux';
import { finalize, takeUntil } from 'rxjs/operators';
import { boolean, object } from 'yup';

import { useEndpointsInAwacsChecker } from '../../../../../components/stage-huge-form/hooks';
import { apiError, NotificationsContext, parseYtApiError } from '../../../../../models';
import { stagesSlice, useStage } from '../../../../../redux';
import { ypApi } from '../../../../../services';

import modalClasses from '../../../../../design/commonModals.module.css';
import classes from './RemoveStageConfirmModal.module.css';

const initialValue = {
   checked: false,
};

type FormValue = typeof initialValue;

const validationSchema = object<FormValue>({
   checked: boolean().oneOf([true], 'You should check this to remove stage'),
});

interface Props extends IModalProps<void> {
   stageId: string;
   projectId: string;
}

export const RemoveStageConfirmModal: React.FC<Props> = React.memo(({ ok, cancel, stageId, projectId }) => {
   const dismounted = useDismounted();
   const notifications = useContext(NotificationsContext);
   const [errorMessage, setErrorMessage] = useState('');

   const [awacs, setAwacs] = useState({
      checking: true,
      isValid: null as boolean | null,
      error: null as ReactNode,
   });

   const [loaderText, setLoaderText] = useState('');

   const { rawStage } = useStage(stageId, true);
   const rawStageRef = useRef(rawStage);

   const endpointsInAwacsChecker = useEndpointsInAwacsChecker(true);
   const dispatch = useDispatch();

   const handleSubmit = useCallback(
      (v: FormValue, helpers: FormikHelpers<FormValue>) => {
         setLoaderText('Stage is removing');

         // временно
         // TODO: реализовать решение в рамках DEPLOY-3695
         ypApi
            .deleteStage(stageId)
            .pipe(
               finalize(() => {
                  setLoaderText('');
                  helpers.setSubmitting(false);
               }),
               takeUntil(dismounted),
            )
            .subscribe(
               () => {
                  ok();
                  dispatch(stagesSlice.actions.deleteFromState(stageId));
               },
               resp => {
                  setErrorMessage(parseYtApiError(resp).shortError);
                  apiError('Stage deletion', resp, notifications);
               },
            );
      },
      [dismounted, dispatch, notifications, ok, stageId],
   );

   useEffect(() => {
      if (!rawStageRef.current) {
         return;
      }

      setLoaderText('Checking used AWACS balancers...');

      endpointsInAwacsChecker(null, rawStageRef.current)
         .pipe(
            finalize(() => setLoaderText('')),
            takeUntil(dismounted),
         )
         .subscribe(checkResult => {
            setAwacs({
               ...checkResult,
               checking: false,
            });
            if (!checkResult.isValid) {
               setErrorMessage('The stage cannot be removed before fix errors');
            }
         });
   }, [dismounted, endpointsInAwacsChecker]);

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   const canBeRemoved = !awacs.checking && awacs.isValid;

   // TODO сделать компонент для форм внутри модалок с собственным "сохранением"
   return (
      <ModalLayout onDismiss={dismiss} title={'Stage deletion'} showFooter={false}>
         <Formik initialValues={initialValue} onSubmit={handleSubmit} validationSchema={validationSchema}>
            {form => (
               <Form className={classes.form}>
                  <p>
                     Caution! You are about to delete a stage: <b>{stageId}</b> in project: <b>{projectId}</b>.
                  </p>

                  <CheckboxField2
                     name={'checked'}
                     label={null}
                     disabled={!canBeRemoved}
                     controlAttrs={{ 'data-e2e': 'RemoveStageConfirmModal:SureCheckbox' }}
                  >
                     Yes, I know what I'm doing, delete <b>{stageId}</b>
                  </CheckboxField2>

                  {!awacs.isValid && <div className={classes.error}>{awacs.error}</div>}

                  <footer className={classes.footer}>
                     {errorMessage && !form.isSubmitting ? <span className={classes.error}>{errorMessage}</span> : null}

                     {loaderText && (
                        <span data-test={'RemoveStageConfirmModal:Loader'}>
                           <Loader inline={true} text={loaderText} />
                        </span>
                     )}

                     <div className={classes.spacer} />

                     <div className={classes.actions}>
                        <Button
                           view={'flat'}
                           className={classes.button}
                           onClick={dismiss}
                           disabled={form.isSubmitting}
                           extraProps={{ 'data-e2e': 'RemoveStageConfirmModal:CancelButton' }}
                        >
                           Cancel
                        </Button>

                        <Button
                           view={'action'}
                           type={'submit'}
                           className={classNames(classes.button, modalClasses.danderButton)}
                           disabled={!form.isValid || form.isSubmitting || !canBeRemoved || !form.values.checked}
                           extraProps={{ 'data-e2e': 'RemoveStageConfirmModal:OkButton' }}
                        >
                           Delete
                        </Button>
                     </div>
                  </footer>
               </Form>
            )}
         </Formik>
      </ModalLayout>
   );
});

RemoveStageConfirmModal.displayName = 'RemoveStageConfirmModal';
