import React, { useMemo } from 'react';

import { Label, Popover, Spin } from '@yandex-cloud/uikit';
import { ButtonLink, formatDate } from '@yandex-infracloud-ui/libs';

import { StatusType } from '../../../../../components/lib';
import { urlBuilder } from '../../../../../models';
import { useConfig } from '../../../../../services';
import { StageDescription } from '../../status/components/StageDescription/StageDescription';
import { StageRevision } from '../../status/components/StageRevision/StageRevision';
import { StageStatusBadge } from '../../status/components/StageStatusBadge/StageStatusBadge';
import { StageXRay } from '../../status/components/StageXRay/StageXRay';
import { useStageStatusView } from '../../status/hooks';
import { StageTurnClusterButton } from '../StageTurnClusterButton/StageTurnClusterButton';
import { useStagePermissions, useStageProblems } from '../../../../../redux';
import { StageProblemsLabel } from './components/StageProblemsLabel/StageProblemsLabel';
import {
   getIssueSummaryFromList,
   IssueLevelColors,
   issueLevels,
   labelThemeByLevel,
} from '../../../../../modules/infra-doctor';

import classes from './StageStatusSummaryLine.module.css';

interface Props {
   stageId: string;
}

export const StageStatusSummaryLine: React.FC<Props> = React.memo(({ stageId }) => {
   const config = useConfig()!;
   const userHasAccessToEdit = useStagePermissions(stageId);
   const { stageStatusViewList } = useStageStatusView({ stageSelectData: [{ stageId }] });
   const [stageStatusView = null] = stageStatusViewList;

   const { revision: statusRevision, updateDate, spec, locations, disabledLocations } = stageStatusView ?? {};

   const clustersForIgnore = useMemo(
      () =>
         Array.from(locations ?? []).filter(
            cluster => !disabledLocations?.has(cluster) && config.isClusterLost(cluster),
         ),
      [config, disabledLocations, locations],
   );

   const { issues, isLoading } = useStageProblems(stageId);
   const { summary: problemSummary } = useMemo(() => getIssueSummaryFromList({ issues }), [issues]);

   return (
      <div className={classes.line}>
         <StageStatusBadge stageId={stageId} type={StatusType.Primary} />
         <Popover
            openOnHover={false}
            placement={['bottom', 'auto']}
            content={
               <>
                  <div>{updateDate && formatDate(updateDate)}</div>
                  <div style={{ fontStyle: 'italic' }}>
                     <StageDescription
                        text={spec?.revisionDescription ?? null}
                        title={`Description for revision ${statusRevision} of stage ${stageId}`}
                     />
                  </div>
                  <br />
                  <ButtonLink to={urlBuilder.stageStatusInfo(stageId)}>See full page</ButtonLink>
               </>
            }
         >
            <Label size={'m'}>
               <StageRevision stageId={stageId} />
            </Label>
         </Popover>
         <Popover
            openOnHover={false}
            placement={['bottom', 'auto']}
            content={
               <>
                  <h3>Stage health</h3>
                  <br />
                  {isLoading ? (
                     <Spin />
                  ) : (
                     <div style={{ display: 'flex', gap: '8px' }}>
                        <Label theme={labelThemeByLevel[problemSummary.summaryLevel]}>
                           <span style={{ color: IssueLevelColors[problemSummary.summaryLevel] }}>
                              {problemSummary.summaryLevel}
                           </span>
                        </Label>
                        {problemSummary.totalCount > 0 && (
                           <>
                              Issues:
                              {issueLevels.map(level => {
                                 const count = problemSummary.countByLevel[level];
                                 return count ? (
                                    <span style={{ color: IssueLevelColors[level] }}>
                                       {count} {level}
                                    </span>
                                 ) : null;
                              })}
                           </>
                        )}
                     </div>
                  )}
                  <br />
                  <h3>XRay</h3>
                  <StageXRay stageId={stageId} />
                  <br />
                  <ButtonLink to={urlBuilder.stageStatusInfo(stageId)}>See full page</ButtonLink>
               </>
            }
         >
            <StageProblemsLabel stageId={stageId} />
         </Popover>
         {userHasAccessToEdit.edit && (
            <>
               {clustersForIgnore.map(cluster => (
                  <StageTurnClusterButton cluster={cluster} stageId={stageId} action={'disable'} />
               ))}
               {Array.from(disabledLocations ?? []).map(cluster => (
                  <StageTurnClusterButton cluster={cluster} stageId={stageId} action={'enable'} />
               ))}
            </>
         )}
      </div>
   );
});
