import React, { useCallback, useMemo, useState } from 'react';

import { DISMISS_REASON, IModalProps, Loader, ModalLayout } from '@yandex-infracloud-ui/libs';
import { Button } from '@yandex-cloud/uikit';
import { useDispatch } from 'react-redux';

import {
   getStages,
   updateStageDisabledClusters,
   useNetworkErrors,
   useRequestControl,
   useStage,
} from '../../../../../redux';

import { stageClusterActionTitles } from './model';
import { ActionState } from '../../../../../redux/hooks/useRequestControl';
import { YpErrorPlate } from '../../../../../components/network';

import classes from './StageTurnClusterConfirmModal.module.css';
import { LostClusterWarningPlate } from '../../../../../components';

interface Props extends IModalProps<void> {
   stageId: string;
   cluster: string;
   action: 'disable' | 'enable';
}

export const StageTurnClusterConfirmModal: React.FC<Props> = React.memo(({ ok, cancel, stageId, cluster, action }) => {
   const { stage } = useStage(stageId);
   const oldDisabledClusterSet: Set<string> = useMemo(() => new Set(stage?.labels?.deploy?.disabled_clusters ?? []), [
      stage?.labels?.deploy?.disabled_clusters,
   ]);
   const existDeployLabel = Boolean(stage?.labels?.deploy);
   const requestKey = `${action}-${cluster}-${stageId}`;

   const [loaderText, setLoaderText] = useState('');

   const dispatch = useDispatch();
   const { actionStates, activate } = useRequestControl([requestKey], {
      onSuccess: () => {
         ok();
         dispatch(getStages({ objectIds: [stageId] }));
      },
      onFinalize: () => {
         setLoaderText('');
      },
   });

   const actionState = actionStates.get(requestKey);
   const isError = actionState === ActionState.Error;

   const errors = useNetworkErrors([requestKey]);
   const error = errors[requestKey];

   const runAction = useCallback(() => {
      setLoaderText(`Cluster ${cluster} will be ${action}d`);

      const newDisabledClusterSet = new Set(oldDisabledClusterSet);
      if (action === 'disable') {
         newDisabledClusterSet.add(cluster);
      } else {
         newDisabledClusterSet.delete(cluster);
      }

      activate(() => {
         dispatch(updateStageDisabledClusters(stageId, Array.from(newDisabledClusterSet), existDeployLabel));
      });
   }, [action, activate, cluster, dispatch, existDeployLabel, oldDisabledClusterSet, stageId]);

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout
         onDismiss={dismiss}
         title={`${stageClusterActionTitles[action]} ${cluster.toUpperCase()}`}
         showFooter={false}
      >
         <LostClusterWarningPlate cluster={cluster} isDisabled={action === 'enable'} />
         <div className={classes.actions}>
            {loaderText && <Loader inline={true} text={loaderText} />}
            <div className={classes.spacer} />

            <Button view={'flat'} onClick={dismiss}>
               Cancel
            </Button>

            <Button view={'action'} onClick={runAction} disabled={actionState === ActionState.Pending}>
               {stageClusterActionTitles[action]}
            </Button>
         </div>
         {isError && error?.error && <YpErrorPlate error={error.error} request={error.request} />}
      </ModalLayout>
   );
});

StageTurnClusterConfirmModal.displayName = 'StageTurnClusterConfirmModal';
