import {
   classNames,
   DISMISS_REASON,
   IApiError,
   IModalProps,
   Loader,
   ModalLayout,
   parseApiError,
   useDismounted,
} from '@yandex-infracloud-ui/libs';
import { Button } from 'lego-on-react';
import React, { useCallback, useState } from 'react';
import { finalize, takeUntil } from 'rxjs/operators';

import modalClasses from '../../../../../../design/commonModals.module.css';
import { ReleaseRule } from '../../../../../../models/ui/releases';
import { ypApi } from '../../../../../../services';
import { ResourceWarning } from './ResourceWarning';

import classes from './RuleRemoveModal.module.css';

interface Props extends IModalProps<any> {
   rule: ReleaseRule;
}

export const RuleRemoveModal: React.FC<Props> = React.memo(({ rule, ok, cancel }) => {
   const dismounted = useDismounted();
   const [apiError, setApiError] = useState('');
   const [isLoading, setIsLoading] = useState(false);

   const handleOk = useCallback(() => {
      setIsLoading(true);
      ypApi
         .removeRule(rule.id)
         .pipe(
            finalize(() => setIsLoading(false)),
            takeUntil(dismounted),
         )
         .subscribe(ok, (resp: IApiError) => {
            setApiError(parseApiError(resp));
         });
   }, [dismounted, ok, rule.id]);

   const dismiss = useCallback(() => cancel(DISMISS_REASON), [cancel]);

   return (
      <ModalLayout title={null} showFooter={false} onDismiss={dismiss}>
         <div>
            <p>
               You are going to remove <b>{rule.id}</b> release rule.
            </p>

            {rule.form ? (
               <div className={classes.resourceWarning}>
                  <ResourceWarning rule={rule} />
               </div>
            ) : null}

            <p>Are you sure you want to remove the release rule?</p>
         </div>

         <footer className={modalClasses.footer}>
            {apiError && !isLoading ? <span className={modalClasses.error}>{apiError}</span> : null}

            <Loader visible={isLoading} inline={true} text={'Rule is removing'} />

            <Button
               theme={'normal'}
               size={'s'}
               view={'default'}
               tone={'default'}
               cls={modalClasses.button}
               onClick={dismiss}
               disabled={isLoading}
            >
               Cancel
            </Button>

            <Button
               theme={'action'}
               size={'s'}
               view={'default'}
               tone={'default'}
               cls={classNames(modalClasses.button, modalClasses.danderButton)}
               onClick={handleOk}
               disabled={isLoading}
            >
               Remove
            </Button>
         </footer>
      </ModalLayout>
   );
});

RuleRemoveModal.displayName = 'RuleRemoveModal';
