import { faPlus, faTimes } from '@fortawesome/pro-regular-svg-icons';
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { YCSelect } from '@yandex-data-ui/common';
import { ExtendedFieldConfig, FieldLayout2, FormButton, useExtendedField } from '@yandex-infracloud-ui/libs';
import React, { useCallback, useState } from 'react';
import { useSelector } from 'react-redux';
import { Link } from 'react-router-dom';

import { LegoButton } from '../../../../../../../../../../_lego';
import { urlBuilder } from '../../../../../../../../../../models';
import { ApprovalPolicyFormParams } from '../../../../../../../../../../models/ui';
import { RootState, selectRulesByStage } from '../../../../../../../../../../redux';

import classes from './ReleaseRuleExclusionsField.module.css';

type ExclusionsFieldProps = ExtendedFieldConfig<any, ApprovalPolicyFormParams['exclusions']['releaseRules']>;

export const ReleaseRuleExclusionsField = React.memo((props: ExclusionsFieldProps & { stageId: string }) => {
   const { readonly, onChange, disabled, field } = useExtendedField(props as ExclusionsFieldProps);
   const { value } = field;

   const { stageId } = props;

   const rulesSelector = useCallback((s: RootState) => selectRulesByStage(s, stageId), [stageId]);
   const rules = useSelector(rulesSelector);

   const ruleItems = rules
      .filter(rule => !value.has(rule.id))
      .map(rule => ({
         value: rule.id,
         title: rule.id,
      }));

   const [selectedRules, setSelectedRules] = useState<string[]>([]);

   const handleAdd = useCallback(() => {
      for (const ruleId of selectedRules) {
         value.add(ruleId);
      }
      onChange(new Set(value));
      setSelectedRules([]);
   }, [onChange, value, selectedRules]);

   const onRemove = useCallback(
      (releaseRuleId: string) => {
         value.delete(releaseRuleId);
         onChange(value);
      },
      [onChange, value],
   );

   return (
      <FieldLayout2 {...props}>
         {Array.from(value.values()).map(releaseRuleId => {
            const url = urlBuilder.releaseRulePage(props.stageId, releaseRuleId);
            return (
               <div key={releaseRuleId} className={classes.row}>
                  <div>
                     <Link to={url}>{releaseRuleId}</Link>
                  </div>

                  <div>
                     <LegoButton
                        theme={'clear'}
                        onClick={() => onRemove(releaseRuleId)}
                        disabled={disabled || readonly}
                     >
                        <FontAwesomeIcon icon={faTimes} />
                     </LegoButton>
                  </div>
               </div>
            );
         })}

         {ruleItems.length > 0 && (
            <YCSelect
               type={'multiple'}
               onUpdate={setSelectedRules}
               value={selectedRules}
               items={ruleItems}
               showSearch={true}
               placeholder={'Select rules'}
            />
         )}

         {selectedRules.length > 0 && (
            <div className={classes.addButton}>
               <FormButton icon={faPlus} onClick={() => handleAdd()} skipLeftSpace={true}>
                  Add {selectedRules.length} rule{selectedRules.length > 1 ? 's' : ''}
               </FormButton>
            </div>
         )}
      </FieldLayout2>
   );
});

ReleaseRuleExclusionsField.displayName = 'ReleaseRuleExclusionsField';
