import { deepClone } from '@yandex-infracloud-ui/libs';
import { Button } from 'lego-on-react';
import { isEqual } from 'lodash';
import React, { useCallback, useEffect, useMemo, useState } from 'react';
import { useDispatch } from 'react-redux';

import { FormPage } from '../../../../../../components';
import { RolesList } from '../../../../../../components/common';
import {
   approvalPolicyActionsArray,
   ApprovalPolicyConverter,
   ApprovalPolicyFormConverter,
   ApprovalPolicyRequestKeys,
   ApprovalPolicyRequestKeysActions,
   approvalPolicyValidationSchema,
} from '../../../../../../models/ui';
import ContainerLoader from '../../../../../../old-code/components/ContainerLoader/ContainerLoader';
import { useApprovalPolicy } from '../../../../../../redux/hooks/useApprovalPolicy';
import {
   deleteApprovalPolicy as deletePolicy,
   fetchApprovalPolicy as fetchPolicy,
   createApprovalPolicy as createPolicy,
   updateApprovalPolicy as updatePolicy,
} from '../../../../../../redux/slices/yp/slices/approvalPolicy';
import { useStage } from '../../../../../../redux/hooks/useStage';
import { useNetworkErrors } from '../../../../../../redux';
import { YpErrorTooltip } from '../../../../../../components/network';
import { NetworkItem } from '../../../../../../redux/slices/network/model';
import { noop } from '../../../../../../utils';
import { useProjectAcl } from '../../../../../../utils/hooks/useProjectAcl';

import { ApprovalPolicyForm } from './components/ApprovalPolicyForm/ApprovalPolicyForm';

import classes from './index.module.scss';

// TODO Why no-shadow?
// eslint-disable-next-line no-shadow
enum PageType {
   loading = 'loading',
   empty = 'empty',
   form = 'form',
   confirmation = 'confirmation',
   delete = 'delete',
}

type DispatchType = (...args: any[]) => Promise<void>;

interface Props {
   stageId: string;
   approvalVisible: boolean;
   bumpApprovalUpdateCount: () => void;
}

// TODO переписать нормально
export const ApprovalPolicy: React.FC<Props> = React.memo(
   ({ stageId, approvalVisible, bumpApprovalUpdateCount }: Props) => {
      const [page, setPage] = useState(PageType.loading);

      const [currentAction, setAction] = useState({ func: noop });

      const dispatch: DispatchType = useDispatch();

      const { stage } = useStage(stageId);
      const projectId = stage?.project?.id ?? '';

      const fetchPolicyRequestKey = ApprovalPolicyRequestKeys.getOne({
         stageId,
         action: ApprovalPolicyRequestKeysActions.FETCH,
      });
      const fetchApprovalPolicy = useCallback(
         () => dispatch(fetchPolicy.withRequestKey(fetchPolicyRequestKey)(stageId)),
         [stageId, dispatch, fetchPolicyRequestKey],
      );

      const requestKeys = approvalPolicyActionsArray.map(action =>
         ApprovalPolicyRequestKeys.getOne({
            stageId,
            action,
         }),
      );

      const errors = useNetworkErrors(requestKeys);
      const errorsList = useMemo(() => Object.values(errors).filter<Partial<NetworkItem>>(Boolean as any), [errors]);

      const createPolicyRequestKey = ApprovalPolicyRequestKeys.getOne({
         stageId,
         action: ApprovalPolicyRequestKeysActions.CREATE,
      });
      const createApprovalPolicy = useCallback(
         spec =>
            dispatch(
               createPolicy.withRequestKey(createPolicyRequestKey)({
                  meta: {
                     id: stageId,
                     stage_id: stageId,
                  },
                  spec,
               }),
            ),
         [stageId, dispatch, createPolicyRequestKey],
      );

      const updatePolicyRequestKey = ApprovalPolicyRequestKeys.getOne({
         stageId,
         action: ApprovalPolicyRequestKeysActions.UPDATE,
      });
      const updateApprovalPolicy = useCallback(
         spec => dispatch(updatePolicy.withRequestKey(updatePolicyRequestKey)(stageId, spec)),
         [stageId, dispatch, updatePolicyRequestKey],
      );

      const deletePolicyRequestKey = ApprovalPolicyRequestKeys.getOne({
         stageId,
         action: ApprovalPolicyRequestKeysActions.DELETE,
      });
      const deleteApprovalPolicy = useCallback(
         () => dispatch(deletePolicy.withRequestKey(deletePolicyRequestKey)(stageId)),
         [stageId, dispatch, deletePolicyRequestKey],
      );

      const { approvalPolicySpec, meta } = useApprovalPolicy(stageId);
      const isLoading = meta.isProcessing;
      const isEmpty = meta.isNotFound;

      const [, bumpRenderVersion] = useState(0);
      const rawAcl = useProjectAcl(projectId);
      const acl = rawAcl || {};
      const stageAcl = (acl.stage_acl || {})[stageId] || {};
      const projectAcl = (acl.project_acl || {})[projectId] || {};
      const projectApprovers = projectAcl.approver || [];
      const projectMandatoryApprovers = projectAcl.mandatory_approver || [];
      const approvers = stageAcl.approver || [];
      const mandatoryApprovers = stageAcl.mandatory_approver || [];

      useEffect(() => {
         bumpRenderVersion(e => e + 1);
      }, [rawAcl]);

      useEffect(() => {
         if (isLoading) {
            return; // сбрасываем тип только при открытии окна
         }
         if (isEmpty) {
            setPage(PageType.empty);
         } else if (!isLoading) {
            setPage(PageType.form);
         }
      }, [
         isEmpty, // сброс при создании политики
         isLoading, // сброс при загрузке политики
         approvalVisible, // сброс при новом открытии всплывающего окна
      ]);

      const roles = (
         <div className={classes.roles}>
            <div>
               <span className={classes.roleTitle}>Regular approvers</span>
               <br />
               <RolesList
                  projectId={projectId}
                  stageId={stageId}
                  accounts={approvers}
                  inheritedAccounts={projectApprovers}
                  role={'approver'}
               />
            </div>
            <div>
               <span className={classes.roleTitle}>Mandatory approvers</span>
               <br />
               <RolesList
                  projectId={projectId}
                  stageId={stageId}
                  accounts={mandatoryApprovers}
                  inheritedAccounts={projectMandatoryApprovers}
                  role={'mandatory_approver'}
               />
            </div>
         </div>
      );

      const loadingMode = useMemo(
         () => () => (
            <div className={classes.page}>
               <ContainerLoader />
            </div>
         ),
         [],
      );

      const emptyMode = useMemo(
         () => () => (
            <div>
               <p>There is no approval policy. Let’s create a new.</p>
               <div className={classes.buttonRow}>
                  <Button
                     theme={'action'}
                     size={'s'}
                     view={'default'}
                     tone={'default'}
                     type={'submit'}
                     onClick={() => setPage(PageType.form)}
                     mix={{ block: classes.createButton }}
                     controlAttrs={{
                        'data-e2e': 'ApprovalPolicy:InitCreating',
                     }}
                  >
                     Create Approval Policy
                  </Button>
               </div>
            </div>
         ),
         [],
      );

      const formInitialValues = ApprovalPolicyFormConverter.getValuesFromSpec(approvalPolicySpec ?? undefined);

      const rawInitialValues = deepClone(formInitialValues);

      const onSubmitForm = useCallback(
         values => {
            // eslint-disable-next-line symbol-description
            const formId = Symbol();
            const func = isEmpty ? createApprovalPolicy : updateApprovalPolicy;
            const action = async () => {
               const spec = ApprovalPolicyFormConverter.getSpecFromValues(values);
               const rawSpec = ApprovalPolicyConverter.getRawSpec(spec);
               func(rawSpec)
                  .then(() => fetchApprovalPolicy())
                  .then(() => {
                     setPage(PageType.form);
                     bumpApprovalUpdateCount(); // пробрасываем наверх информацию об обновлении
                  })
                  .catch(() => {
                     setPage(isEmpty ? PageType.empty : PageType.form);
                  });
            };
            action.formId = formId;

            if (isEmpty) {
               // новую политику создаём сразу
               setPage(PageType.loading);
               action();
            } else {
               // для существующей требуем подтверждение
               setAction({ func: action as any });
               setPage(PageType.confirmation);
            }
         },
         [isEmpty, createApprovalPolicy, updateApprovalPolicy, fetchApprovalPolicy, bumpApprovalUpdateCount],
      );

      const formMode = () => (
         <FormPage
            customLayout={true}
            cancelButtonText={'Cancel'}
            subTitle={`${isEmpty ? 'Create' : 'Update'} approval policy`}
            submitButtonText={'Save'}
            title={'Approval Policy'}
            initialValues={formInitialValues}
            validationSchema={approvalPolicyValidationSchema}
            onSubmit={async values => {
               onSubmitForm(values);
            }}
            onCancel={noop}
         >
            {form => (
               <div className={classes.formContainer}>
                  <ApprovalPolicyForm stageId={stageId} />
                  <hr className={classes.divider} />
                  {roles}
                  <div className={classes.buttonRow}>
                     {!isEmpty && (
                        <Button
                           theme={'normal'}
                           size={'s'}
                           view={'classic'}
                           tone={'default'}
                           text={'Delete'}
                           onClick={() => setPage(PageType.delete)}
                           controlAttrs={{
                              'data-e2e': 'ApprovalPolicy:Delete',
                           }}
                        />
                     )}
                     <Button
                        theme={'action'}
                        size={'s'}
                        view={'default'}
                        tone={'default'}
                        text={isEmpty ? 'Create' : 'Update'}
                        disabled={!isEmpty && isEqual(rawInitialValues, form.values)}
                        onClick={() => onSubmitForm(form.values)}
                        mix={{ block: classes.updateButton }}
                        controlAttrs={{
                           'data-e2e': `ApprovalPolicy:${isEmpty ? 'Create' : 'Update'}`,
                        }}
                     />
                  </div>
               </div>
            )}
         </FormPage>
      );

      const deleteMode = () => (
         <div>
            <p>Are you sure you want to delete approval policy?</p>
            <div className={classes.buttonRow}>
               <Button
                  theme={'normal'}
                  size={'s'}
                  view={'default'}
                  tone={'default'}
                  text={'Cancel'}
                  onClick={() => {
                     setPage(PageType.form);
                  }}
               />
               <Button
                  theme={'action'}
                  size={'s'}
                  view={'default'}
                  tone={'default'}
                  text={'Delete'}
                  onClick={async () => {
                     setPage(PageType.loading);
                     // Нашел небольшую багу: Если при удалении прилетает ошибка - зависает бесконечный лоадинг
                     await deleteApprovalPolicy();
                     fetchApprovalPolicy();
                  }}
                  controlAttrs={{
                     'data-e2e': 'ApprovalPolicy:ConfirmDeleting',
                  }}
               />
            </div>
         </div>
      );

      const confirmationMode = useMemo(
         () => () => (
            <div>
               <p>
                  Are you sure you want to update approval policy? The approval statuses of the deploy tickets will not
                  be recalculated.
               </p>
               <div className={classes.buttonRow}>
                  <Button
                     theme={'normal'}
                     size={'s'}
                     view={'default'}
                     tone={'default'}
                     text={'Cancel'}
                     onClick={() => {
                        setAction({ func: noop });
                        setPage(isEmpty ? PageType.empty : PageType.form);
                     }}
                  />
                  <Button
                     theme={'action'}
                     size={'s'}
                     view={'default'}
                     tone={'default'}
                     text={'Accept'}
                     onClick={() => {
                        setPage(PageType.loading);
                        currentAction.func();
                        setAction({ func: noop });
                     }}
                     controlAttrs={{
                        'data-e2e': 'ApprovalPolicy:ConfirmUpdating',
                     }}
                  />
               </div>
            </div>
         ),
         [isEmpty, currentAction],
      );

      const content = {
         [PageType.loading]: loadingMode,
         [PageType.form]: formMode,
         [PageType.empty]: emptyMode,
         [PageType.confirmation]: confirmationMode,
         [PageType.delete]: deleteMode,
      }[page]();

      return (
         <div data-e2e={'ApprovalPolicy:Body'}>
            <div>{isLoading ? loadingMode() : content}</div>
            {errorsList.map(
               ({ error, request }) =>
                  error && (
                     <div className={classes.networkError}>
                        <YpErrorTooltip error={error} request={request} />
                     </div>
                  ),
            )}
         </div>
      );
   },
);

ApprovalPolicy.displayName = 'ApprovalPolicy';
